// ScrApp.cpp: implementation of the CScrApp class.  This file contains 
//  all of the main logic for scredit2.
// Mark R. Riordan (MRR)  December 2005
//
// This program source code and executable are released under the MIT X11 license.

/*===== Correction History ====================================================
 * MRR - 2006-04-16  0.4.5.0
 *   -- Implement the following settings in scredit2.ini: 
 *      FontFace, FontSize, NTextblocks (default to 254).
 *   -- Make some symbol changes to allow > 46 textblocks.
 *   -- Fix a bug in modlm which didn't always adjust the left margin.
 *   -- Fix a display bug which could draw beyond col 80 in on bottom line.
 *   -- Increase macro wait time to 2500 ms.
 *   -- Add Compiled HTML documentation.
 * MRR - 2006-04-15  0.4.4.0
 *   -- Add checke().
 *   -- Parse \n in macros.
 *   -- Implement delete char and Toggle Insert Mode.
 *   -- Implement the status line (the 25th line on the screen) & show Insert.
 * MRR - 2006-04-09  0.4.3.0
 *   -- Implement cursor left and right in terminal mode.
 *   -- Implement adjustment of left margin in terminal mode.
 *   -- Draw thin line around active portion of window.
 * MRR - 2006-04-08  0.4.2.0
 *   -- Implement a settings file, scredit2.ini, in the Application Data 
 *      directory and implement the
 *   -- Implement macros.  The only use of macros so far is the command line 
 *      -m macname, specifies characters to be sent after connecting.
 *   -- Implement a default connection, to be filled in in the Connect dialog.
 *   -- Implement trrbs.
 *   -- Implement expansion of tabs received in terminal mode (not in orig).
 */

#include "stdafx.h"
#include "ScrApp.h"

#define TIMER_CURSOR 1
#define TIMER_MACRO  2

extern CScrUtil ScrUtil;
extern HINSTANCE hInst;
extern HWND hWndMain;
extern char szVersion[];

CScrApp *pThisApp=NULL; // Pointer to single instance of CScrApp.  Needed by static members. 

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CScrApp::CScrApp()
{
   pThisApp = this;
}

void CScrApp::Initialize()
{
	m_Port = 23;
	m_bConnected = FALSE;
	m_bSysErr = FALSE;
	m_MacroCharsSent = 0;
   m_bSendCmdsAtStartup = FALSE;
	m_bCursorOn = FALSE;
	m_hFontNormal = NULL;
	m_MarginTop = 5;
	m_MarginLeft = 5;
	m_LinesOnScreen = SC_PHSCN;
	m_CharsPerLine = 80;
	m_PixelsInLine = 0;
	refForeground = RGB(0, 255, 0);
	refBackground = RGB(0, 0, 0);
	refPenBorder = RGB(0, 160, 0);
	varinit();

	m_Settings.ReadSettings();
   InitializeMappings();
   init();
}

CScrApp::~CScrApp()
{

}

void CScrApp::syserr(const char *msg, const char *file, int linenum)
{
   char szmsg2[1024];
   _snprintf(szmsg2, sizeof(szmsg2), "scredit2 detected an error:\r\n\r\n%s\r\n\r\nin %s at line %d",
      msg, file, linenum);
	m_bSysErr = TRUE;
   int retcode = MessageBox(m_hwnd, szmsg2, "scredit2 internal error - press Cancel to quit", MB_OKCANCEL | MB_ICONEXCLAMATION);
   if(IDCANCEL == retcode) {
      exit(666);
   }
}

//=====  General-purpose utility functions  ==============================
void CScrApp::fill(Byte8 *dest, int nbytes, Byte8 mybyte) 
{
   memset(dest, mybyte, nbytes);
}

void movei(Byte8 *source, Byte8 *dest, int len)
{
   memcpy(dest, source, len);
}

/*--- function moved -----------------------------------
*
*        MOVE BYTES, USING DECREASING ADDRESSES
*
*         MOVE BYTES AROUND IN MEMORY, STARTING AT AN ADDRESS
*         AND GOING BACKWARD.  THIS ROUTINE IS THEREFORE USEFUL
*         FOR FORWARD MOVES IN MEMORY, AS IT DOES NOT DESTROY
*         THE LATTER PART OF THE SOURCE AREA WHEN THE SOURCE AND
*         DESTINATION AREAS OVERLAP.
*
*         ENTRY    sourcelwa CONTAINS THE LAST ADDR OF THE SOURCE
*                  destlwa   CONTAINS THE LAST ADDR OF THE DESTINATION
*                  nbytes    CONTAINS THE NUMBER OF BYTES TO BE MOVED
*/
void moved(Byte8 *sourcelwa, Byte8 *destlwa, int nbytes)
{
   while(nbytes > 0) {
      *(destlwa--) = *(sourcelwa--);
      nbytes--;
   }
}

/*--- function moveq ------------------------------
*       MOVE BYTES FORWARD OR BACKWARD
*
*         ENTRY    fwa CONTAINS THE FWA OF A STRING OF BYTES
*                  TO MOVE.
*                  lwap1  CONTAINS THE LWA+1 OF THE STRING OF BYTES.
*                  howfar CONTAINS HOW FAR THE STRING IS TO BE
*                    SHIFTED, AND THE DIRECTION OF THE SHIFT.
*                    ITS MAGNITUDE IS HOW FAR TO SHIFT.
*                    IF IT IS NEGATIVE, THE SHIFT IS DOWNWARD,  TOWARD
*                    DECREASING CM ADDRESSES, ELSE THE SHIFT IS UPWARD.
*
*         calls    moved   to do forward byte moves
*                  movei   to do backward byte moves
*
*         method   fairly straightforward
*/
void moveq(Byte8 *fwa, Byte8 *lwap1, int howfar)
{
   if(howfar > 0) {
      // MOVE THE BYTES FORWARD.  compute the source lwa, dest lwa,
      // and length for "moved".
      Byte8 *destlwa = lwap1 + howfar - 1;
      int nbytes = (int)(lwap1 - fwa);
      Byte8 *sourcelwa = lwap1 - 1;
      moved(sourcelwa, destlwa, nbytes);
   } else {
      // MOVE THE BYTES DOWNWARD IN MEMORY.  compute the source fwa,
      // destination fwa, and byte count for "movei".
      int nbytes = (int)(lwap1 - fwa);
      Byte8 *destfwa = fwa + howfar;
      movei(fwa, destfwa, nbytes);
   }
}

//--- function movelno ------------------------------------------
//  Move a line number.
void movelno(Byte8 *source, Byte8 *dest)
{
   movei(source, dest, LN_LENNO);
}

/*--- function copybuf ------------------------------------------
*         copy an unpacked buffer of ascii characters.  buffer must
*         be ln.txlen chars long.  (they all are.)
*
*         entry    source = fwa of source
*                  dest = fwa of dest
 */
void copybuf(Byte8 *source, Byte8 *dest)
{
   movei(source, dest, LN_TXLEN);
}

//--- function movemsg ------------------------------------------
// MOVE ZERO-BYTE-TERMINATED STRING
void movemsg(Byte8 *source, Byte8 *dest)
{
   while(*source) {
      *(dest++) = *(source++);
   }
}

// Do an unsigned compare of 2 buffers of bytes.
// Exit:  returns 0 if bufs are equal, 
//                1 if buf1 > buf2
//               -1 if buf1 < buf2
int compbyt(Byte8 *bytes1, Byte8 *bytes2, int len)
{
   int j;
   for(j=0; j<len; j++) {
      if(bytes1[j] > bytes2[j]) {
         return 1;
      } else if(bytes1[j] < bytes2[j]) {
         return -1;
      }
   }
   return 0;
}

int complno(Byte8 *linenum1, Byte8 *linenum2)
{
   return compbyt(linenum1, linenum2, LN_LENNO);
}

/*--- function scn2lnb ---------------------------------
*         SCAN TO THE LAST NON-BLANK CHAR ON A LINE)
*
*         ENTRY    HL=ADDR OF FWA OF LINE
*                  DE=LWA+1 OF LINE IMAGE
*
*         EXIT     BC=# OF CHARS IN LINE
*                  DE=LWA OF LINE
*                  HL=ADDR OF FWA OF LINE
*
*         calls    none
*
*         method   repeat
*                    decrement char_ptr
*                  until char_ptr^ <> ' '
*                  if char_ptr is not before beg of line then
*                    char_count := char_ptr - line fwa + 1
*                  else
*                    char_count := 0
*                    char_ptr := fwa of line
*                  endif
*/
int scn2lnb(Byte8 *fwa, Byte8 *lwap1)
{
	int nchars;
	do {
		lwap1--;
	} while(' '==*lwap1 && lwap1 > fwa);
	nchars = lwap1 - fwa + 1;
	return nchars;
}

/*--- function GetNibble -----------------------
 *  Entry:  aryBytes is an array of bytes.
 *          nibble   is the nibble # we want.  0=top 4 bits of aryBytes[0];
 *                   1=bottom 4 bits of aryBytes[1], 2=top 4 bits of aryBytes[1], etc.
 *  Exit:   Returns the desired nibble.
 */
Byte8 GetNibble(Byte8 *aryBytes, int nibble)
{
   int mybyte = aryBytes[nibble/2];
   if(nibble&1) {
      return mybyte & 0xf;
   } else {
      return (mybyte>>4) & 0xf;
   }
}

/*--- function lno2asc -----------------------------
*         CONVERT INTERNAL LINE # TO ASCII
*
*         ENTRY    HL CONTAINS THE FWA OF AN INTERNALLY-FORMATTED
*                    LINE #
*                  DE CONTAINS THE FWA OF A REGION OF MEMORY TO STORE
*                    THE LINE #
*
*         EXIT     [DE]-+12 Contain an ASCII representation of the
*                    line #.  Leading and trailing zeros have been
*                    replaced with blanks and the decimal point has
*                    been replaced with a blank whenever possible.
*         This routine has been rewritten from scratch and bares no
*         resemblance to the original.
*/
void lno2asc(Byte8 *linenum, char *pASCII)
{
   int j, nibble, idxout=0;
   BOOL bLeadingZeros=TRUE;
   for(j=0; j<6; j++) {
      nibble = GetNibble(linenum, j);
      if(0==nibble) {
         if(bLeadingZeros && 5!=j) {
            pASCII[idxout++] = ' ';
         } else {
            pASCII[idxout++] = '0';
         }
      } else {
         bLeadingZeros = FALSE;
         pASCII[idxout++] = '0'+nibble;
      }
   }
   idxout = 12;
   BOOL bTrailingZeros = TRUE;
   for(j=11; j>5; j--) {
      nibble = GetNibble(linenum, j);
      if(0==nibble) {
         if(bTrailingZeros) {
            pASCII[idxout--] = ' ';
         } else {
            pASCII[idxout--] = '0';
         }
      } else {
         bTrailingZeros = FALSE;
         pASCII[idxout--] = '0'+nibble;
      }      
   }
   pASCII[6] = bTrailingZeros ? ' ' : '.';
}

std::string lno2StdString(Byte8 *linenum)
{
   char ASCII[24];
   lno2asc(linenum, ASCII);
   size_t idxstart=0, idxend=12;
   while(' '==ASCII[idxstart]) idxstart++;
   while(' '==ASCII[idxend]) idxend--;
   return std::string(ASCII+idxstart, idxend-idxstart+1);
}

/*--- function decbnz ---------------------------------
 *  decrement a byte in memory if it's not already zero.
 *  Entry:	pByte	points to a byte.
 *  Exit:	That byte has been decremented if isn't not already zero.
 */
void decbnz(Byte8 *pByte)
{
	if(*pByte) (*pByte)--;
}

//=====  Initialization functions  ======================================
//---  See also InitializeMappings, near the bottom.

/*--- function CScrApp::varinit ---------------------
 */
void CScrApp::varinit() {
   fg_insr = 0;
   fg_mful = 0;
   fg_tful = 0;
   fg_hlti = 0;
   fg_srti = 0;
   fg_bsci = 0;
   fg_mode = MD_TRM;
   fg_modo = fg_mode;
   fg_ccmd = 0;
   fg_rlin = 0;
   fg_clik = 0;
   fg_lkls = 0;
   fg_glst = 0;
   fg_lnex = 0;
   fg_buch = 0;
   fg_cken = 0;
   fg_ck1l = 0;
   fg_gfrs = 0;
   fg_tdsb = 0;
   fg_mdsb = 0;
   fg_sdsb = 0;
   fg_ktim = 0;
   fg_kchr = 0;
   fg_stay = 0;
   fg_edit = 0;
   fg_bldf = 0;
   fg_mddr = 0;
   fg_mdmd = 0;
   fg_upcs = 0;
   fg_wrdp = 0;
   fg_keyp = 0;
   fg_esc = 0;
   fg_dll = 0;
   fg_nomm = 0;
   fg_clmb = 0;
   fg_smen = 0;
   fg_rkey = 0;
   fg_cobl = FALSE;
   fg_stat = 0;
   fg_stmd = 0;
   fg_ltsc = 0;
   fg_blos = 0;
   fg_fdb = 0;
   fg_rdy = 0;
   fg_ndry = 0;
   fg_mans = 0;
   fg_cewc = 0;
   fg_dndb = 0;
   fg_bwms = 0;
   fg_dnds = 0;
   fg_abow = 0;
   fg_nobd = 0;
   fg_usbl = 0;
   fg_init = 0;
   fg_ver = 0;
   fg_dbug = 0;
   fg_dfpb = 0;
   fg_tcfp = 0;
   fg_oeow = 0;
   fg_crlc = 0;
   fg_olhm = 0;
   fg_newv = 0;
   fg_ftfl = 0;
   fg_soh = 0;
   fg_stx = 0;
   fg_dle = 0;
   fg_end = 0;
   fg_ftmd = 0;
   fg_ftb1 = 0;
   fg_tail = 0;
   fg_ftcn = 0;
   fg_ftrn = 0;
   fg_ftc6 = 0;
   fg_fts7 = 0;
   fg_ftml = 0;
   fg_sens = 0;
   fg_frpm = 0;
   fg_fspm = 0;
   iirbyte = 0;
   fg_com1 = 0;
   fg_com2 = 0;
   fg_brk = 0;
   fg_mdst = 0;
   fg_fse = 0;
   fg_v2ec = 0;
   fg_v2nl = 0;
   fg_v2q = 0;
   fg_v2ca = 0;
   fg_v2hl = 0;
   fg_v2in = 0;
   fg_nkap = 0;
   fg_v2g = 0;
   fg_scst = 0;
   fg_dcaw = 0;


   ct_cmch = 0;
   ct_rtln = 0;
   ct_cken = 0;
   ct_okln = 0;
   ct_okmx = 0;
   ct_dlld = 0;
   ct_rev = 0;
   ct_bked = 0;
   ct_blok = 0;
   ct_bktr = 0;
   ct_trmn = 0;
   ct_trmx = 0;
   ct_cstm = 0;
   ct_trln = 0;
   ct_edln = 0;
   ct_ftdp = 0;
   ft_pvch = 0;
   ct_debs = 0;
   ct_mxnd = 0;
   ct_nlnd = 0;
   ct_lf = 0;
   ct_wtlf = 0;
   ct_lldv = 0;
   ct_tick = 0;
   ct_v2v = 0;

   m_bKeyHandled = FALSE;

}

void CScrApp::inittrm()
{
   fg_mode = MD_TRM;
   if(fg_clmb) {
      // We need to clear edit mode
      //! flesh out
   }
   fg_rlin = FALSE;  // no more 'waiting for mainframe'
   fg_cobl = FALSE;  // we say cursor not on bot line
   trbotm();         // so trbotm will initialize clb.
   dsscrn();
   if(fg_init) {
      // This is not program initialization, but rather we are reentering
      // terminal mode during normal operation.
      //! flesh out 
   }
}

void CScrApp::init()
{
   int j;
   // screen
   for(j=0; j<SC_PHSCN; j++) {
      sprintf((char *)screen+j*SC_LNLEN, "This is line %2d", j);
   }
   botline = screen + 23*SC_LNLEN;
   msgl = botline;
   statusl = screen + 22*SC_LNLEN;
   debugl = screen + 21*SC_LNLEN;

   // Textblocks
   bk_ntxt = m_Settings.GetNTextblocks();
   int nbytes_blocks = bk_ntxt*BLK_LENG;
   blocks = (Byte8 *) malloc(nbytes_blocks);
   fill(blocks, nbytes_blocks, 0);
   for(j=0; j<BLK_NUMB; j++) bk_uset[j] = MD_FREE;  // textblock usage table

   ct_trmx = bk_ntxt - CT_MNEDB;  // maximum # of blocks for term mode
   ct_trmn = CT_MNTRB;           // default minimum # of blocks

   blnkscr();

   ct_okmx = CT_NOKLN;  // DEFAULT VALUE--ALTERED BY 'SAFETY'

   fill(bf_cpyl, LN_TXLEN, ' ');
   fill(bf_del, LN_TXLEN, ' ');
   fill(clb_buf, sizeof(clb_buf), ' ');
   clb_buf[sizeof(clb_buf)-1] = 0;
   clb_zero = 0;
   tabset_zero = 0;
   fill(tabset1, sizeof(tabset1), 0);
   fill(tabset2, sizeof(tabset1), 0);

   fill(bf_inln, sizeof(bf_inln), ' ');
   bf_inln[sizeof(bf_inln)-1] = 0;

   fg_mode = MD_TRM; 
   ct_mxnd = CT_DFDND;  // default number of lines to wait before forcing display in T mode

   // Initialize tabs to 9, 17, 25, etc.
   fill(sc_tabs, sizeof(sc_tabs), 0);
   for(j=8; j<=72; j+=8) sc_tabs[j] = 1;

   sc_v2bm = SC_PHSCN-1; // set bottom scrolling margin

   // Initialize textblock index, setting the first 3 entries to End markers.
   idxfwa1[0] = MK_INEND;  // beg of edit mode index
   Byte8* pidxent = idxfwa1;
   nxtidx(pidxent);
   *pidxent = MK_INEND; // end of edit/beg of term mode,
   nxtidx(pidxent);
   *pidxent = MK_INEND; // end of term mode
   idxlwa1 = edindex + sizeof(edindex);   // LWA+1 of index
   tridxpt = pidxent;   // This is currently the first terminal mode entry.

   blb_fkc = MK_NOKYC;  // never any keyboard chars in the build-line buffer.

   newibf();   // put a blank line in term mode index

   // Put a version message on the terminal mode screen.
	char szStartup[100];
	int nchars=_snprintf(szStartup, sizeof(szStartup), "scredit2 %s of %s %s  http://scredit2.sourceforge.net",
		szVersion, __DATE__,__TIME__);
   movemsg((Byte8 *)szStartup, bf_inln);
   addibf();
   newibf();

   inittrm();
   fg_init = TRUE;   // Initialization is complete.
}

//--- function CScrApp::AddBCD -------------------
// Entry:   bcd1, bcd2 are 4-bit integers 0-9.
//          carry is 0 or 1.
// Exit:    Returns the sum mod 10.
//          carry is the new carry flag, 0 or 1.
Byte8 CScrApp::AddBCD(Byte8 bcd1, Byte8 bcd2, Byte8 &carry)
{
   Byte8 result = bcd1 + bcd2 + carry;
   carry = 0;
   if(result > 9) {
      carry = 1;
      result -= 10;
   }
   return result;
}

// Add two line numbers
// Entry:   linenum1, linenum2   are line numbers in BCD format.
// Exit:    result      is the sum.
void CScrApp::addlno(Byte8 *linenum1, Byte8 *linenum2, Byte8 *result)
{
   int j;
   Byte8 carry=0;
   for(j=LN_LENNO-1; j>=0; j--) {
      Byte8 bottom = AddBCD(linenum1[j]&0xf, linenum2[j]&0xf, carry);
      Byte8 top = AddBCD((linenum1[j]>>4) & 0xf, (linenum2[j]>>4) & 0xf, carry);
      result[j] = (top<<4) | bottom;
   }
}

/*--- function clb2ibf ------------------------------
*         copy the clb to the bottom line buffer
*
*         copy the current line buffer to the buffer used for the
*         bottom line in terminal mode (when the cursor is off
*         the bottom line).
*
*         entry    none
*
*         exit     bf.inln and associated variables (pt.inln and
*                  ct.bfkc) contain the information in the clb.
*
*         calls    copybuf to copy a buffer */
void CScrApp::clb2ibf()
{
   copybuf(clb_buf, bf_inln);
   pt_inln = clb_col;  // copy cursor position
   ct_bfkc = clb_fkc;  // copy char ordinal of first keybd char
}

/*--- function clb2int --------------------
*   Current line buffer format TO INTERNAL LINE
*
*         ENTRY    inbytes = fwa of a line in a special format:  the
*                    first ln.hdlen bytes are the exact bytes
*                    that should go in the line header (except
*                    for the length byte, which can have any value),
*                    and the remaining ln.txlen bytes are the
*                    characters in "straight" ascii.
*                    The line ends in at least 2 blanks.
*                    Note that the current line buffer (CLB) is
*                    kept in this format.
*
*         EXIT     bf.misc HAS THE COMPLETE, INTERNALLY-FORMATTED LINE.
*
*         calls    movei   to move a string of bytes
*
*         method   (*  Copy the line header directly.
*                      Converting the text portion is essentially
*                      a byte-by-byte copy, complicated by the fact
*                      that consecutive identical characters are
*                      packed into a special two-byte sequence.
*                      This requires setting flags within the copy
*                      loop, indicating that a duplicated sequence
*                      is being processed, and then paying attention
*                      to these flags. *)
*
*                  move line header from beg of source to beg of dest
*                  oldchar := 0
*                  numdup := 0
*                  packidx := 0
*                  loop
*                    if source^ = 0 then goto endline
*                    if source^ = oldchar then
*                      increment numdup
*                      increment source
*                    else
*                      oldchar := source^
*                      (* check for end of duplicated sequence *)
*                        if numdup <> 0 then
*                          intline[packidx] := numdup + 1
*                          numdup := 0
*                          set high bit of intline[packidx-1]
*                          increment packidx
*                        endif
*                      (* add character to line *)
*                        intline[packidx] := oldchar
*                        source := source + 1
*                    endif
*                  endloop
*                  endline:
*                    linelength := packidx - 1
*/
void CScrApp::clb2int(Byte8 *inbytes)
{
   ScrUtil.DebugOut("clb2int: %-2.2x%-2.2x%-2.2x.%-2.2x%-2.2x%-2.2x=%s", inbytes[0], inbytes[1], inbytes[2], inbytes[3], inbytes[4], inbytes[5], inbytes+LN_HDLEN);
   // Copy line header.  rely on the fact that the
   //     clb variables are set up in the same order as the fields
   //     in a line header.
   movei(inbytes, bf_misc, LN_HDLEN);
 
   /*         CONVERT THE LINE.  RELY ON THE FACT THAT THERE ARE TWO
   *         BLANK BYTES AT THE END OF CLB.BUF, PLUS A ZERO BYTE.
   *         THE ZERO BYTE SIGNIFIES THE END OF THE BUFFER; THE TWO
   *         BLANKS GUARANTEE THAT THERE WILL BE A DUPLICATED BLANK
   *         SEQUENCE AT THE END OF THE LINE.  IT'S GROSS, BUT
   *         IT'S FAST!
   *         IN THE LOOP,
   *         C = # OF DUPED CHARS--USUALLY ZERO
   *         B = PREV CHAR
   *         DE POINTS TO THE NEXT CLB CHAR
   *         HL POINTS TO THE PLACE TO PUT THE NEXT INTERNAL CHAR,
   *           OR, IN THE CASE OF DUPED CHARS, THE DUP COUNT BYTE.
   */
   Byte8 oldchar=0;
   int numdup=0, packidx=0;
   Byte8 *dest=bf_misc+LN_HDLEN;
   Byte8 *source = inbytes+LN_HDLEN;

   do {
      if(!*source) break;
      if(*source == oldchar) {
         numdup++;
         source++;
      } else {
         oldchar = *source;
         if(numdup) {
            dest[packidx] = numdup + 1;
            numdup = 0;
            dest[packidx-1] |= 0x80;
            packidx++;
         }
         dest[packidx++] = oldchar;
         source++;
      }
   } while(true);
   bf_misc[OF_LNLEN] = LN_HDLEN+packidx-1;
}

//=====  Textblock routines  ========================================

/*--- function CScrApp::bkn2adr -------------------------------------
*         get block address from block number
*
*         used by the "blkadr" macro to find the address of a
*         textblock from its number.
*
*         entry    a = the textblock number
*
*         exit     hl = fwa of actual textblock
*/
Byte8 *CScrApp::bkn2adr(unsigned int blocknum)
{
   return blocks + (BLK_LENG * blocknum);
}

/*--- function blkadr ----------------------
**        BLKADR -- COMPUTE ADDRESS OF TEXTBLOCK FROM INDEX ENTRY
*
*         BLKADR   (NO PARAMETERS)
*
*         INPUT:   pIdxEnt points to an ENTRY IN TEXTBLOCK INDEX
*         OUTPUT:  Returns ADDRESS OF BEG OF TEXTBLOCK
*
*         calls:   bkn2adr to compute block addr from block number
*/
Byte8 *CScrApp::blkadr(Byte8 *pIdxEnt)
{
   unsigned int blocknum = pIdxEnt[OF_INBKN];  // Extract block number from textblock entry
   return bkn2adr(blocknum);
}

/*--- function begidx ----------------------------------------
 *  find the beg of the textblock index for the given mode.
 *  Exit:   Returns the FWA of the textblock index for that mode.
 */
Byte8 *CScrApp::begidx(TypProgMode mode)
{
   Byte8 * pbeg=NULL;
   if(MD_MSUE == mode) {
      pbeg = edindex;
   } else {
      pbeg = tridxpt;
   }
   return pbeg;
}

// Increment pidxent to the next entry in the textblock index.
// Exit: Return TRUE if next entry is a textblock, else FALSE if end-of-index.
BOOL CScrApp::nxtidx(Byte8 *&pidxent)
{
   pidxent += IDX_ENTR;
   return (MK_INEND != *pidxent);
}

/*--- function prvidx ------------------------------------
 *  skip to previous textblock index entry.
 *  Exit:   Returns TRUE if we found the previous entry, else FALSE
 *             if we went past the beginning.
 *          pidxent  is the previous entry
 */
BOOL CScrApp::prvidx(Byte8 *&pidxent)
{
   pidxent -= IDX_ENTR;
   return (MK_INBEG != *pidxent);
}

/*--- function freeblk -----------------------------------------
*         obtain a free textblock
*
*         entry    mode =      the mode the textblock is for (md.xxx)
*                  bk.nfre =FWA OF LIST OF TEXTBLOCK #'S (ONE PER BYTE)
*                           to not allocate, presumably because they're
*                           BEING USED FOR SOMEthING ELSE IMPORTANT.
*                           list is terminated by mk.endls byte.
*                           the list is ignored if anything but an edit
*                           mode block winds up getting allocated.
*                           The list must not be so restrictive that
*                           neither the first nor the last block
*                           in edit mode can be allocated.  (If an
*                           edit mode block winds up being required.)
*                  ct.trmx  is the maximum number of textblocks
*                           in terminal mode.
*                  ct.trmn  is the minimum number of textblocks
*                           in terminal mode.  The minimum number
*                           of blocks must be small enough so
*                           that one screen's worth of text, plus
*                           one block, is left for edit mode.
*                           Otherwise, the algorithm might not work.
*
*         exit     TextblockNum = textblock # found
*                  Returns 1 if operation was successful, else
*                    zero if could not get a block.
*
*         calls    delblk to delete a currently-used block
*                  nxtidx to skip to next index entry
*                  srchbyt to scan list of forbidden blocks
*                  FINDBLK   To find text block a line is in
*
*         uses     all
*
*         method   scan usage table for free block
*                  if there was a free block then
*                    return
*                  else
*                    if block is for edit mode then
*                      if # of terminal mode blocks > minimum then
*                        free first terminal mode block
*                      else
*                        allocate edit mode block (see below)
*                      endif
*                    else  (* block is for terminal mode *)
*                      if # of terminal mode blocks >= maximum then
*                        free first terminal mode block
*                      else
*                        allocate edit mode block (see below)
*                      endif
*                    endif
*                  endif
*
*                  to allocate an edit mode block:
*
*                    IF FIRST EDIT MODE BLOCK IS not forbidden then
*                      free first edit mode block
*                    else
*                      if last edit mode block is not in list then
*                        free last edit mode BLOCK
*                      else
*                        return 'cannot allocate block'
*                      endif
*                    endif
*                    return 'success'
 */
BOOL CScrApp::freeblk(TypProgMode mode, int &TextblockNum)
{
   BOOL bFound = FALSE;
	BOOL bFreeTermBlock=FALSE;
   
   for(TextblockNum = 0; TextblockNum<bk_ntxt; TextblockNum++) {
      if(MD_FREE == bk_uset[TextblockNum]) {
         bFound = TRUE;
         break;
      }
   }
   if(!bFound) {
		TextblockNum = -1;
		if(MD_MSUE == mode) {
			//! flesh out
			CALL_SYSERR("freeblk: MSUEditor mode not yet implemented");
		} else {
			// The block is needed for terminal mode.
			if(ct_bktr >= ct_trmx) {
				// We are already at the limit for terminal mode blocks,
				// so free an existing terminal mode block.
				bFreeTermBlock = TRUE;
			}
		}
		if(bFreeTermBlock) {
			TextblockNum = tridxpt[OF_INBKN];
		} else {
			CALL_SYSERR("freeblk: free edit mode not implemented");
		}
		delblk(TextblockNum);
		bFound = TRUE;
   }
	if(TextblockNum >= bk_ntxt || TextblockNum < 0) {
		CALL_SYSERR("freeblk: found bad textblock number");
	}
   return bFound;
}

/*--- function findblk -----------------------------------------
*         FIND THE BLOCK A LINE SHOULD RESIDE IN
*
*         ENTRY    linenum POINTS TO A 6-BYTE LINE #
*                  mode = applicable program mode.
*
*         EXIT     Returns TRUE if the block was found (for that mode),
*                    in which case
*                  pBlkIndexEntry points TO THE TEXTBLOCK INDEX
*                    ENTRY THAT CORRESPONDS TO WHERE THE LINE IS OR
*                    should be.
*                  a = false if no block exists for that mode
*
*         calls    begidx  to find the beginning of the index
*                  nxtidx  to find the next index entry
*                  prvidx  to find the previous index entry
*                  complno to compare line numbers
*
*         METHOD -
*                  (*  Most lines being searched for are near the
*                      end of the index, so we start by skipping
*                      to the end.  Then we back up index entry
*                      by index entry until we encounter an entry
*                      whose first line number is <= the line
*                      number being searched for.
*                      If no such entry exists, just return the
*                      first entry in the index.  *)
*
*                  position index_ptr just before first index entry
*                  repeat
*                    skip index_ptr to next entry
*                  UNTIL INDEX_PTR positioned beyond end of index
*                  REPEAT
*                    Skip INDEX_PTR to previous entry
*                    IF before beginning of index GOTO TOOFAR
*                  UNTIL INDEX_PTR^.LINE_# <= INPUT_LINE_#
*                  RETURN "SUCCESS"
*
*                  TOOFAR:
*                  Skip INDEX_PTR to next entry
*                  IF beyond end of index THEN
*                    RETURN "FAILURE"
*                  ELSE
*                    RETURN "SUCCESS"
*                  ENDIF
*/
BOOL CScrApp::findblk(Byte8 *linenum, TypProgMode mode, Byte8 *&pBlkIndexEntry)
{
   BOOL bFound=FALSE;
   pBlkIndexEntry = begidx(mode);
   // Skip to before beginning
   prvidx(pBlkIndexEntry);
   do {
      // nothing
   } while(nxtidx(pBlkIndexEntry));
   // pBlkIndexEntry points to the end of the textblock index.
   // Now move the index pointer back through the index until
   // we go past the beginning or encounter an index entry
   // that is <= the one we're looking for.
   BOOL bGotPrev;
   do {
      bGotPrev = prvidx(pBlkIndexEntry);
      if(!bGotPrev) break;
   } while(complno(pBlkIndexEntry, linenum) > 0);
   if(bGotPrev) {
      bFound = TRUE;
   } else {
      // We hit the beginning of the textblock index without
      // finding the block.
      if(nxtidx(pBlkIndexEntry)) {
         // We found the first block in the index, so return success.
         bFound = TRUE;
      }
   }

   return bFound;
}

/*--- function findln ------------------------------
*         FIND A LINE IN A TEXTBLOCK SOMEWHERE
*
*         ENTRY    linenum POINTS TO A 6-BYTE LINE NUMBER.
*                  mode = THE MODE OF THE LINE BEING LOOKED FOR.
*
*         EXIT     Returns  false  IF THE LINE WAS not FOUND.
*                  pBlkIndexEntry is A POINTER TO THE TEXTBLOCK INDEX ENTRY
*                    THAT CORRESPONDS TO WHERE THE LINE IS OR SHOULD BE.
*                  pLineInBlock is THE ADDRESS OF WHERE THE LINE STARTS
*                    (OR SHOULD START) IN THE TEXTBLOCK.
*
*         calls    findblk to find the right textblock
*                  complno to compare line numbers
*                  syserr  to report system error
*
*         method   (*  find the textblock the line is in.
*                      skip through the lines in the block, looking
*                      for the input line number, or one that is
*                      greater.  If greater, we've found the spot,
*                      but must return "line not found".  *)
*
*                  IF line number begins with end of text marker
*                    RETURN with A = FALSE - Line not found
*                  ENDIF
*                  find correct textblock for line
*                  if block cannot be found then
*                    return "failure"
*                  else
*                    line_ptr := beginning of block
*                    loop
*                      if line_ptr points to too small a line # then
*                        if line_ptr's line length is zero then
*                          system error
*                        else
*                          advance line_ptr to next line in block
*                        endif
*                      else if the line numbers match then
*                        return "found"
*                      else  (* we've gone too far *)
*                        return "not found"
*                      endif
*                    endloop
*/
BOOL CScrApp::findln(Byte8 *linenum, TypProgMode mode, 
    Byte8 *&pBlkIndexEntry, Byte8 * &pLineInBlock)
{
   //ScrUtil.DebugOut("findln: Entered with linenum=%s", lno2StdString(linenum).c_str());
   BOOL bFound=FALSE;
   pBlkIndexEntry = NULL;
   pLineInBlock = NULL;
   // Don't bother to look if linenum starts with end of text marker.
   if(MK_INEND != *linenum) {
      if(findblk(linenum, mode, pBlkIndexEntry)) {
         // We found the textblock where this line should be.
         pLineInBlock = blkadr(pBlkIndexEntry) + BLK_HEAD;
         // Skip though the lines in the block until we find one with line# > linenum.
         while(complno(pLineInBlock, linenum) < 0) {
            Byte8 len = pLineInBlock[OF_LNLEN];
            //ScrUtil.DebugOut("findln: skipped past line %s; len=%d", lno2StdString(pLineInBlock).c_str(), len);
            if(len < LN_HDLEN) {
               CALL_SYSERR("findln: bad line length");
            }
            pLineInBlock += len;
         }
         if(complno(pLineInBlock, linenum) == 0) {
            // We found the line.
            bFound = TRUE;
         }
         // Regardless of whether the line was found, pLineInBlock is where it
         // should be.
      }
   }

   return bFound;
}

/*--- function findlcm ----------------------------------------------
*         find a line w/ given number in current mode)
*
*         find the index entry address and address in textblock memory
*         of a line with a given line number.  restrict the search
*         to textblocks in the current mode.
* 
*         ENTRY    linenum POINTS TO A 6-BYTE LINE NUMBER.
*
*         EXIT     Returns  false  IF THE LINE WAS not FOUND.
*                  pBlkIndexEntry is A POINTER TO THE TEXTBLOCK INDEX ENTRY
*                    THAT CORRESPONDS TO WHERE THE LINE IS OR SHOULD BE.
*                  pLineInBlock is THE ADDRESS OF WHERE THE LINE STARTS
*                    (OR SHOULD START) IN THE TEXTBLOCK.
*/
BOOL CScrApp::findlcm(Byte8 *linenum, Byte8 *&pBlkIndexEntry, Byte8 * &pLineInBlock)
{
   return findln(linenum, fg_mode, pBlkIndexEntry, pLineInBlock);
}

/*--- function cntln -----------------------------------
*         counts the number of lines in all the blocks in a
*         particular index (terminal or edit mode).
*
*         entry    pBlkIndexEntry = fwa of appropriate index
*
*         exit     Returns # of lines in that index
*
*         CALLS    NXTIDX  TO skip to next index entry
*                  prvidx  to skip to previous index entry
*
*         method   back up index pointer
*                  count = 0
*                  while not end of index do
*                    skip to next entry
*                    count = count + lines in this entry
*                  endwhile
 */
int CScrApp::cntln(Byte8 *pBlkIndexEntry)
{
   int nlines=0;
   prvidx(pBlkIndexEntry);
   do {
      if(!nxtidx(pBlkIndexEntry)) break;
      nlines += pBlkIndexEntry[OF_INNLN];
   } while(true);
   return nlines;
}

/*--- function bookln ----------------------------------
*         do bookkeeping for add/delete of a line
*
*         this routine is called whenever a line or block is
*         added or deleted.  it updates the variables whose
*         values may change due to the addition or deletion.
*         It also recovers from the presumably impossible
*         condition of the CLB disappearing from textblock memory.
*
*         entry    Textblock, CLB, and index are set up as usual.
*
*         exit     the following variables have been updated:
*                  ct.edln  (count of lines in edit mode)
*                  ct.trln  (count of lines in term mode)
*                  fg.ltsc  (whether there's a whole screen in t mode)
*                  ds.trtp  (line # of top line in t mode)
*                  the clb has been replaced with a nearby line
*                    if the textblock line corresponding to its
*                    previous contents has somehow been wiped.
*
*         calls    begidx  to find beginning of index
*                  cntln   to count lines for a given mode
*                  nearln  to find a nearby line
*                  int2clb to unpack a line
*                  xmitclb to send the clb to mainframe
*
*         method   (* variables for edit mode *)
*                    count # of lines in edit mode
*                  (* variables for terminal mode *)
*                    count # of lines in term mode
*                    less_than_whole_screen = (count < # lines on scrn)
*                  (* variables for current mode *)
*                    look for copy of clb (current mode) in textblock
*                    if not found, then
*                      if edit mode then
*                        send clb to mainframe if it's altered.
*                      endif
*                      clb = line near former clb
*                    endif
 */
void CScrApp::bookln()
{
   // update variables for edit mode
   Byte8 *pLineInBlock, *pBlkIndexEntry = begidx(MD_MSUE);
   ct_edln = cntln(pBlkIndexEntry);
   // update variables for terminal mode
   pBlkIndexEntry = begidx(MD_TRM);
   ct_trln = cntln(pBlkIndexEntry);

   // Set flag for 'less than whole screen'.
   fg_ltsc = (ct_trln < SC_PHSCN);

   // UPDATE VARIABLES FOR CURRENT MODE (edit or term).
   if(MD_STA != fg_mode) {
      if(!findlcm(clb_lno, pBlkIndexEntry, pLineInBlock)) {
         // clb is no longer in textblock memory, so clean out clb
         // and put a new line in it.
         if(MD_MSUE == MD_MSUE) {
            //! flesh this out.
         }
         if(nearln(clb_lno, fg_mode, pBlkIndexEntry, pLineInBlock)) {
            //! int2clb(
         }
      }
   }
}

/*--- function delblk ----------------------------------------
*         Delete a block from index and usage table)
*
*         Deallocate a textblock by marking it as "free" in the
*         usage table and (for Terminal and Edit modes) removing it
*         from the textblock index.
*
*         ENTRY -  TextblockNum = textblock number to delete
*
*         EXIT -   We expect the block to be found, and successfully
*                    deleted.
*                  IF the block could not be found, this routine
*                    declares a system error and does not return.
*
*         USES -   All registers
*
*         CALLS -  MOVEQ   To manipulate textblock index
*                  NXTIDX  To skip to next index entry
*                  BEGIDX  To get the beginning of the index
*                  SYSERR  To indicate system error and halt machine
*                  BOOKLN  To do accounting.
*
*         METHOD - Save current mode from usage tbl entry for block
*                  IF block number = MK.endls (end flag - unallocated)
*                    Just RETURN
*                  ELSE (not end of index flag)
*                    Zero the usage table entry for the block
*                    IF mode of textblock was MD.ft (file transfer)
*                      Just RETURN -  (Only in CPU = 8086 versions)
*                    ELSE    (Not file transfer)
*                      indexptr := beg of index for correct mode
*                      WHILE indexptr.block# <> this block number do
*                        skip to next index entry
*                        IF end of index then
*                          System error
*                        ENDIF
*                      ENDWHILE
*                      Move bottom of index back to cover deleted entry
*                      IF previous mode of block was "edit" then
*                        decrement edit mode block count
*                        reposition terminal mode fwa index point
*                      ELSE
*                        decrement terminal mode block count
*                      ENDIF
*                    ENDIF
*                  ENDIF */
void CScrApp::delblk(Word32 TextblockNum)
{
   // MK.endls is a marker for end of index and is preset
   // in some block pointer variables before a block is
   // is actuallly allocated for that particular use.
	ScrUtil.DebugOut("delblk called for block %d", TextblockNum);
   if(MK_ENDLS != TextblockNum) {
      TypProgMode mode = bk_uset[TextblockNum];
      bk_uset[TextblockNum] = MD_FREE;
      if(MD_FT != mode) {
         Byte8 *pBlkIndexEntry = begidx(mode);
         // Search the index for an entry that corresponds to the
         // block # being deleted.
         BOOL bFound = TRUE;
         do {
            if(TextblockNum == pBlkIndexEntry[OF_INBKN]) {
               bFound = TRUE;
               break;
            }
         } while(nxtidx(pBlkIndexEntry));
         if(!bFound) {
            CALL_SYSERR("delblk: block not found");
         }
         // now, delete this entry in the index by moving all of the
         // entries after it back by one entry.
         nxtidx(pBlkIndexEntry); // move to next entry
         moveq(pBlkIndexEntry, idxlwa1, -IDX_ENTR);

         // decrement the appropriate textblock count.
         // also, if the block deleted was an edit mode entry,
         // the terminal mode index fwa is now one slot too far
         // down (since the index has moved backwards), so adjust it.
         if(MD_MSUE == mode) {
            ct_bked--;  // decrement MSU EDITOR count
            prvidx(tridxpt);  // skip the terminal mode index fwa back by one index entry
         } else {
            ct_bktr--;  // decrement terminal mode count
         }
         bookln();
      }
   }
	DebugDump("delblk end");
}

/*--- function dellnb ----------------------------------
*         DELETE A LINE IN A TEXTBLOCK
*
*         ENTRY    linenum POINTS TO THE LINE # OF THE LINE TO DELETE
*                  mode = the mode of the line to delete
*
*         EXIT     Returns TRUE IF THE LINE WAS SUCCESSFULLY
*                  DELETED, ELSE FALSE.
*
*         calls    findln  to find a line
*                  moveq   to move a string of bytes
*                  movelno to move a line number
*                  bookln  to do textblock bookkeeping
*
*         method   find line in textblock
*                  if line found ok then
*                    move bytes after line backwards on top of line
*                    adjust count of bytes in textblock
*                    adjust "first line number in block" index field
*                    decrement count of lines in block
*                    if count of lines is now zero then
*                      delete this textblock
*                    endif
*                    call bookln to do accounting
*                  endif */
BOOL CScrApp::dellnb(Byte8 *linenum, TypProgMode mode)
{
   BOOL bDeleted=FALSE;
   Byte8 *pBlkIndexEntry=NULL, *pLineInBlock=NULL;
   if(findln(linenum, mode, pBlkIndexEntry, pLineInBlock)) {
      // Move the bytes from the beg of the next line to the end
      // of the block backwards in memory, overwriting this line.
      Byte8 *pBegOfBlock = blkadr(pBlkIndexEntry);
      int lenBlock = *(Word16 *)(BLK_OFFSET_ACTIVE_BYTES+pBegOfBlock);
      Byte8 *pLWAp1 = pBegOfBlock+lenBlock;
      int lenLine = pLineInBlock[OF_LNLEN];
      Byte8 *pNextLine = pLineInBlock+lenLine;
      moveq(pNextLine, pLWAp1, -lenLine);

      // line deleted.  now adjust counts.
      // adjust count of bytes in textblock.
      *(Word16 *)(BLK_OFFSET_ACTIVE_BYTES+pBegOfBlock) -= lenLine;

      // adjust the "line # of first line in block" field in the index.
      Byte8 *pFirstLineInBlock = pBegOfBlock+BLK_HEAD;
      if(MK_INEND != *pFirstLineInBlock) {
         movelno(pFirstLineInBlock, pBlkIndexEntry);
      }

      // adjust # of lines in block.  free textblock if no more lines.
      Byte8 new_num_lines = --(pBlkIndexEntry[OF_INNLN]);
      if(0 == new_num_lines) {
         delblk(pBlkIndexEntry[OF_INBKN]);
      }
      bookln();
      bDeleted = TRUE;
   }

   return bDeleted;
}

/*--- function splittr ---------------------------------
*         make a new textblock for terminal mode.
*
*         when adding lines to a textblock in terminal mode, we
*         do not have to go through the complicated measures needed
*         in edit mode (splitting the block into two, etc.) when
*         a textblock overflows.  instead, we just allocate a new
*         one and leave the textblock that would have overflowed
*         alone.  this allows for higher textblock memory utilization
*         in terminal mode.
*
*         entry    pBlkIndexEntry = addr of entry in textblock index that
*                    corresponds to the block being "split".
*                  mode = the mode of the new block
*                  ln_adno = fwa of the line number being added
*                    (and therefore of the new block).
*
*         exit     a new empty block starting with ln.adno has been
*                    allocated.
*
*         calls    addblk  to allocate and initialize a block
*
*         method   simple 
*/
void CScrApp::splittr(Byte8 *pBlkIndexEntry, TypProgMode mode)
{
#if 0
	static int nSplittrs=0;
	char szbuf[132];
	sprintf(szbuf, "Entered splittr time %d", ++nSplittrs);
   DebugDump(szbuf);
#endif
   // delete any current copies of the line first.
   dellnb(ln_adno, mode);

   // set up the calling sequence to "addblk".
   bk_nfre[0] = MK_ENDLS;  // OK to free any block.
   addblk(ln_adno, mode);
   DebugDump("Leaving splittr");
}

/*--- function split -----------------------------------
*         SPLIT A TEXTBLOCK
*
*         ENTRY    pBlkIndexEntry CONTAINS THE ADDRESS OF THE ENTRY IN THE
*                    TEXTBLOCK INDEX THAT CORRESPONDS TO THE TEXTBLOCK
*                    THAT NEEDS TO BE SPLIT.
*                  mode = program mode for new textblock.
*
*         EXIT     THE TEXTBLOCK HAS BEEN SPLIT IN THE MIDDLE,
*                  AND THE SECOND HALF HAS BEEN PLACED IN A NEW
*                  TEXTBLOCK.  IF NECESSARY, AN EXISTING TEXTBLOCK
*                  HAS BEEN FREED UP FOR USE BY THE NEW TEXTBLOCK.
*
*         calls    splittr to split a terminal mode block
*                  addblk  to add a textblock to this mode
*                  movei   to move a string of bytes
*
*         method   (*  if in terminal mode, make "splittr" do the work.
*                      Otherwise, determine the split point, and
*                      adjust textblock and index fields to reflect
*                      only the first half of the block.  Get
*                      another block, put the second half of
*                      the block in it, and set up the new textblock
*                      and index fields appropriately.  *)
*
*         method   if in terminal mode then
*                    call splittr to do the work
*                  else
*                    line_ptr := addr of first line in this block
*                    split_point := line_ptr + split increment
*                    lines_in_first_half := 0
*                    repeat
*                      increment lines_in_first_half
*                      line_ptr := line_ptr^.length + line_ptr
*                    until  line_ptr > split_point
*                    split_point := line_ptr
*                    oldlength := textblock.length
*                    textblock.length := blockfwa - split_point
*                      + trailer_length
*                    bytes_to_copy := oldlength - textblock.length
*                    num_of_new_lines :=index[block#].numlines -
*                      lines_in_first_half
*                    index[block#].numlines := lines_in_first_half
*                    call addblk to find and set up a new textblock
*                    copy bytes_to_copy bytes from split_point
*                      to new textblock
*                    index[newblock#].firstline := line # from line
*                      at split_point
*                    index[newblock#].numlines := num_of_new_lines
*                    add textblock trailer to end of new textblock
*                  endif
 */
void CScrApp::split(Byte8 *pBlkIndexEntry, TypProgMode mode)
{
   if(MD_TRM == mode) {
      splittr(pBlkIndexEntry, mode);
   } else {
      //! flesh out
   }
}

/*--- function addblk ----------------------------------
*         allocate and initialize a new textblock
*
*         pick a suitable textblock (based on the input parameters)
*         and allocate and initialize it.
*
*         entry    pFirstLineNum = fwa of line # of first line in block
*                  mode = desired mode of block
*                  bk.nfre = list of block #'s that may not be
*                    allocated.  this list may not be so restrictive
*                    that addblk cannot allocate any block.
*                    list terminated by mk.endls byte.
*
*         exit     hl = addr of index entry for this new block.
*
*         calls    freeblk to select a free block
*                  findblk to find the right place in the index
*                  moveq to move the index around
*                  nxtidx to skip to next index entry
*
*         method   call freeblk to make a free block
*                  call findblk to find where entry should be added
*                  adjust index pointer from 'findblk' if nec.
*                  open up a space in the index for new entry
*                  fill in line # and block # fields of entry
*                  zero other fields of entry
*                  mark the block as assigned to appropriate mode
*                  if mode = edit then
*                    increment count of edit mode blocks
*                    adjust terminal mode index fwa pointer
*                  else
*                    increment count of terminal mode blocks
*                  endif
 */
Byte8 *CScrApp::addblk(Byte8 * pFirstLineNum, TypProgMode mode)
{
   Byte8 *pBlkIndexEntry=NULL;
   int TextblockNum;
   if(!freeblk(mode, TextblockNum)) {
		CALL_SYSERR("addblk: freeblk could not free a block");
	}
	ScrUtil.DebugOut("addblk: freeblk found block %d", TextblockNum);
   findblk(pFirstLineNum, mode, pBlkIndexEntry);
   // If the found entry is less than the desired line #, skip to next entry.
   if(complno(pBlkIndexEntry, pFirstLineNum) < 0) {
      nxtidx(pBlkIndexEntry);
   }
   // Open up space in textblock index where entry should go.
   Byte8 *plwap1 = edindex + sizeof(edindex) - IDX_ENTR;
   moveq(pBlkIndexEntry, plwap1, IDX_ENTR);
   // Fill in next textblock entry.
   fill(pBlkIndexEntry, IDX_ENTR, 0); // zero entire entry.
   movelno(pFirstLineNum, pBlkIndexEntry);
   pBlkIndexEntry[OF_INBKN] = (Byte8)TextblockNum;

   // do some textblock bookkeeping not related to the index.
   // set up the textblock's initial byte count and place
   // the end-of-textblock marker at the beginning of the block.
   // reconcile the terminal mode fwa pointer if necessary.
   Byte8 *pBegTextblock = bkn2adr(TextblockNum);
   // There are 3 bytes in textblock initially (header + terminator)
   *(Word16 *)(pBegTextblock+BLK_OFFSET_ACTIVE_BYTES) = 1+BLK_HEAD;
   pBegTextblock[BLK_HEAD] = MK_BKEND; // End of data byte

   // place the new mode of the block in the usage table.
   bk_uset[TextblockNum] = mode;

   // update the appropriate block count.
   if(MD_MSUE == mode) {
      ct_bked++;  // Increment # of MSU EDITOR mode blocks.
      // move terminal index fwa ahead one entry to account for previous 'moveq'.
      nxtidx(tridxpt);
   } else {
      ct_bktr++;  // Increment # of terminal mode blocks.
   }

   return pBlkIndexEntry;
}

/*--- function addln --------------------------------
*         ADD A LINE TO A TEXT BLOCK
*
*         ENTRY    HL       POINTS TO A LINE IMAGE, PROPERLY FORMATTED,
*                           WITH THE LINE # IN THE FIRST 6 BYTES.
*                  a =      MODE OF LINE TO BE ADDED.
*                  ct.bked  is the number of edit mode blocks.
*                  ct.bktr  is the number of terminal mode blocks.
*
*         EXIT     THE LINE HAS BEEN ADDED TO THE APPROPRIATE TEXTBLOCK.
*                  THE FOLLOWING ACTIONS HAVE BEEN TAKEN IF NECESSARY:
*                  - CURRENT COPY OF LINE DELETED.
*                  - TEXTBLOCK SPLIT & ANOTHER ONE CREATED.
*                  - PREVIOUSLY MEMORY-RESIDENT TEXTBLOCK ERASED
*                    TO MAKE ROOM FOR NEW TEXTBLOCK
*
*         calls    syserr  to indicate a system error
*                  findln  to find a line in a textblock
*                  split   to split a textblock
*                  addln   to add a line after a split
*                  moveq   to move a string of bytes
*                  movei   to move a string of bytes
*                  complno to compare two line numbers
*                  bookln  to do textblock-related bookkeeping
*
*         method   if no textblocks exist in this mode then
*                    add a textblock to this mode
*                  endif
*                  if addition of this new line will overflow block
*                    if possible
*                      set variables to add to beg of next block
*                    else
*                      call split
*                      call addln recursively
*                      return
*                    endif
*                  endif
*                  if no old copy of line exists then
*                    increment count of lines in this block
*                  endif
*                  move bytes after where line will go to accomodate ln
*                  compute new blocksize
*                  copy new line into its place in the textblock
*                  if new line number < first line of this block then
*                    update "first line number" field in index
*                  endif
*                  call bookln to do textblock accounting
*/
void CScrApp::addln(Byte8 *packed_line, TypProgMode mode)
{
   ln_adno = packed_line;
   bk_nfre[0] = MK_ENDLS;
   int nblocks_this_mode = 0;
   if(MD_MSUE == mode) {
      nblocks_this_mode = ct_bked;
   } else if(MD_TRM == mode) {
      nblocks_this_mode = ct_bktr;
   } else {
      CALL_SYSERR("addln: bad program mode");
   }

   // If there aren't any textblocks for this mode, add one.
   if(0==nblocks_this_mode) {
      addblk(packed_line, mode);
   }

   int linelen = packed_line[OF_LNLEN];

   Byte8 *pBlkIndexEntry=NULL, *pLineInBlock=NULL, *pStartMove=NULL;
   BOOL bFound = findln(packed_line, mode, pBlkIndexEntry, pLineInBlock);
   int sizeChange=linelen;
   if(bFound) {
      int oldlen = pLineInBlock[OF_LNLEN];
      sizeChange = linelen - oldlen;
   }

   // sizeChange is the # of extra bytes (net change) required by this line.
   // Will this new line fit in the current textblock?
   Byte8 *pTextBlock = blkadr(pBlkIndexEntry);
   Word16 nActiveBytes = *((Word16 *) (pTextBlock + BLK_OFFSET_ACTIVE_BYTES));
   int nNewBytes = ((int)nActiveBytes) + sizeChange;
   if(nNewBytes > BLK_LENG) {
      // Textblock will have to be split.
      //! This is much simplified.  Original code tries to put in next block if possible.
      split(pBlkIndexEntry, mode);
      // Now that textblock is split, call ourself recursively.
      addln(packed_line, mode);
      return;
   }
   
   if(bFound) {
      // WE ARE DOING A LINE REPLACE, SO WE WANT TO MOVE BYTES STARTING
      // FROM THE BEG OF THE LINE AFTER THE OLD COPY OF THIS LINE.
      int nSizeOld = pLineInBlock[OF_LNLEN];
      pStartMove = pLineInBlock + nSizeOld;      
   } else {
      // THIS IS AN ADD, NOT A REPLACE, SO INCREMENT THE # OF
      // LINES IN THE INDEX ENTRY.
      pBlkIndexEntry[OF_INNLN]++;
      // ALSO, SINCE NO COPY OF THIS LINE EXISTS, INDICATE THAT
      // WE HAVE TO MOVE BYTES (FORWARD) STARTING FROM WHERE THIS
      // NEW LINE IS DESTINED.
      pStartMove = pLineInBlock;
   }

   // Move the bytes in the textblock.
   moveq(pStartMove, pTextBlock+nActiveBytes, sizeChange);

   // Update # of active bytes in block
   *((Word16 *) (pTextBlock + BLK_OFFSET_ACTIVE_BYTES)) += sizeChange;

   // COPY THE LINE INTO ITS RIGHTFUL PLACE IN THE TEXTBLOCK.
   movei(packed_line, pLineInBlock, linelen);

   // If MSU EDITOR mode, set flag that we have lines now.
   if(MD_MSUE == mode) {
      fg_lnex = TRUE;
   }

   // TEST TO SEE IF THE LINE BEING ADDED COMES BEFORE THE
   // PREVIOUS FIRST LINE IN THE BLOCK.  IF SO, UPDATE THE
   // INDEX ENTRY.
   if(complno(packed_line, pBlkIndexEntry) < 0) {
      // Set the text block entry's lowest line #.
      movei(packed_line, pBlkIndexEntry, LN_LENNO);
   }
   bookln();
}

/*--- function addibf ---------------------------------------
*
*         add the terminal mode bottom line buffer to a textblock.
*
*         entry    The "clb" variables contain the current line
*                    buffer information.
*
*         exit     none
*
*         calls    clb2int to convert line to packed format
*                  addln   to add it to a textblock
*
*         method   convert to packed format
*                  add packed line to textblock
*/
void CScrApp::addibf()
{
   clb2int(botbuf);  // convert to packed format to bf_misc
   addln(bf_misc, MD_TRM); // add to a textblock
}

/*--- function addlncm ----------------------------
*         add a line to a textblock in the current mode)
*
*         entry    packed_line = fwa of a packed line image to add
*                  fg.mode = the mode of the line
*
*         exit     the line has been added to edit mode, if the
*                    current mode is edit, else to terminal mode.
*
*         calls    addln   to do all the work
*
*         method   simple
 */
void CScrApp::addlncm(Byte8 *packed_line)
{
   TypProgMode mymode = (MD_MSUE == fg_mode) ? MD_MSUE : MD_TRM;
   addln(packed_line, mymode);
}

/*--- function addclb -------------------------------
*         UPDATE ALL COPIES OF THE LINE IN THE CLB
*
*         ENTRY    THE CLB HAS BEEN ALTERED, SO WE NEED TO UPDATE
*                  ALL COPIES OF IT (IN CM, and/or MSU EWFILE)
*
*         EXIT     THE CURRENT CONTENTS OF THE CLB HAVE BEEN SENT
*                  TO WHEREEVER NECESSARY.
*
*         calls    clb2int to convert the clb to internal format
*                  addlncm to add a line to textblock in current mode
*                  xmitclb to send the clb to the mainframe
*
*         method   update the textblock copy
*                  if in edit mode then
*                    send the line to the mainframe in nnn=text format
*                  endif
 */
void CScrApp::addclb()
{
   clb2int(clb_lno);
   addlncm(bf_misc);
   //! flesh out for MSU EDITOR mode.
}

/*--- function addlbtm -----------------------------------------
*         add a line to the bottom of terminal mode
*
*         entry    The line is in either the bottom line buffer
*                    or the current line buffer.
*                  fg.cobl indicates whether the cursor is on the
*                    bottom line (and hence which buffer the line
*                    is in).
*                  fg.ltsc indicates whether the screen was full
*                    last time we displayed it in terminal mode.
*
*         exit     The line has been added to textblock storage
*                    for terminal mode, and has been cleared for
*                    the next line that will be received.
*
*         CALLS    ADDCLB  TO PUT THE CLB IN A TEXTBLOCK
*                  ADDIBF  TO PUT THE BOTTOM LINE IN A TEXTBLOCK
*                  NEWIBF  TO CREATE A NEW BOTTOM LINE
*                  IBF2CLB TO MOVE THE BOTTOM LINE TO THE CLB
*                  FINDLN  TO FIND A LINE
*                  NXTLNB  TO SKIP TO THE NEXT LINE
*                  SYSERR  TO DECLARE A SYSTEM ERROR
*                  DSSCRN  TO DISPLAY THE SCREEN
*                  COMPLNO TO COMPARE LINE NUMBERS
*                  MOVELNO TO MOVE A LINE NUMBER
*
*         METHOD   CLEAR DESTRUCTIVE BACKSPACE COUNT
*                  IF CURSOR IS ON BOTTOM LINE THEN
*                    ADD CLB TO TEXTBLOCKS
*                  ELSE
*                    ADD BOTTOM LINE TO TEXTBLOCKS
*                  ENDIF
*                  CREATE EMPTY BOTTOM LINE BUFFER
*                  IF CURSOR ON BOTTOM LINE THEN
*                    MOVE BOTTOM LINE TO CLB
*                  ENDIF
*                  IF (SCREEN IS FULL) AND (CURSOR IS NOT AT TOP OF
*                    SCREEN) THEN
*                    MOVE TOP-OF-SCREEN POINTER TO NEXT LINE DOWN
*                    IF NO NEXT LINE THEN
*                      DECLARE SYSTEM ERROR
*                    ENDIF
*                  ENDIF
 */
void CScrApp::addlbtm()
{
   ct_debs = 0;
   Byte8 bScreenNotFullOld = fg_ltsc;
   //ScrUtil.DebugOut("addlbtm: fg_ltsc=%d", fg_ltsc);
   if(fg_cobl) {
      // Cursor is on bottom line.
      addclb();   // PUT THE CLB IN A TEXTBLOCK
   } else {
      addibf();
   }
   // CLEAR OUT THE BOTTOM LINE BUFFER
   newibf();
   if(fg_cobl) {
      ibf2clb(); // Cursor is on bottom line, so MAKE THE CLB REFLECT THE BOT LINE
   }
   // IF THE SCREEN IS FULL, AND THE CURSOR IS NOT AT THE TOP OF THE SCREEN,
   // THEN MOVE THE SCREEN UP BY ONE LINE BY MOVING THE
   // 'TOP OF SCREEN' POINTER *DOWN* BY ONE LINE.
   if(!bScreenNotFullOld && 0!=complno(ds_tpln, clb_lno)) {
      Byte8 *pBlkIndexEntry, *pLineInBlock;
      // Find top of screen.
      findln(ds_tpln, MD_TRM, pBlkIndexEntry, pLineInBlock);
      // move to next line.
      if(!nxtlnb(pBlkIndexEntry, pLineInBlock)) {
         CALL_SYSERR("addlbtm: No next line");
      }
      movelno(pLineInBlock, ds_tpln);
   }
}

/*--- function newibf  ----------------------
*         blank out the terminal mode bottom line buffer,
*         initialize associated variables, and add the blank buffer
*         to a textblock.
*
*         entry    none
*
*         exit     ln.bttr has been incremented
*                  bf.inln, pt.inln, ct.bfkc
*                    have been initialized.
*
*         calls    fill    to blank out a buffer
*                  addibf  to add the buffer to a textblock
*                  addlno  to increment a line number
*
*         method   create a fresh new empty buffer
*                  convert to packed format & add to textblock
*/
void CScrApp::newibf()
{
   static Byte8 bcdone[LN_LENNO] = {0,0,0,0,0,1};

   fill(bf_inln, LN_TXLEN, ' ');
   pt_inln = 0;   // initialize cursor position
   ct_bfkc = MK_NOKYC;  // no keyboard chars yet

   addlno(ln_bttr, bcdone,ln_bttr);
   addibf();   // add new buffer to textblock
}

/*--- function prvlnb -----------------------------------
*         FIND THE LINE PREVIOUS TO A GIVEN LINE.  LOOK IN
*         TEXTBLOCKS AS NECESSARY, BUT DO NOT ASK MSU EDITOR.
*
*         ENTRY    pLineInBlock POINTS TO A LINE IN A TEXTBLOCK
*                  pBlkIndexEntry POINTS TO THE INDEX ENTRY FOR THAT LINE
*
*         EXIT     pLineInBlock POINTS TO THE LINE PREVIOUS TO THE GIVEN
*                    LINE.  IF NO SUCH LINE EXISTS, HL IS AS IT WAS
*                    ON INPUT.
*                  pBlkIndexEntry POINTS TO THE INDEX ENTRY FOR THE LINE POINTED
*                    TO BY HL.
*                  Returns TRUE IF A PREVIOUS LINE WAS FOUND,
*                    ELSE FALSE.
*
*         calls    prvidx  to back up one index entry
*
*         method   if not at beginning of textblock then
*          midblock: repeat
*                      decrement line_ptr
*                    until line_ptr^ = 0 (* zero byte in line header *)
*                    back up line_ptr to beginning of line header
*                  else
*                    skip backwards one index entry
*                    if previous entry existed then
*                      line_ptr := end of new textblock
*                      goto midblock
*                    else
*                      RETURN failure
*                    endif
*                  endif
*                  RETURN success
*/
BOOL CScrApp::prvlnb(Byte8* &pBlkIndexEntry, Byte8 *&pLineInBlock)
{
   BOOL bFound = FALSE;
   Byte8 *pBlkIndexEntrySave=pBlkIndexEntry;
   Byte8 *pLineInBlockSave = pLineInBlock;
   // Figure out whether the line is the first in its block.
   Byte8 *pFirstInBlock = BLK_HEAD + blkadr(pBlkIndexEntry);
   if(pFirstInBlock == pLineInBlock) {
      // Back up to end of previous block.
      if(prvidx(pBlkIndexEntry)) {
         Byte8 *pBegOfBlock = blkadr(pBlkIndexEntry);
         // Skip to end of prev block.
         int len = *(Word16 *)(BLK_OFFSET_ACTIVE_BYTES+pBegOfBlock);
         pLineInBlock = pBegOfBlock+len;
      } else {
         // SORRY, THE LINE WAS THE FIRST ONE IN ANY TEXTBLOCK.
         pBlkIndexEntry = pBlkIndexEntrySave;
         pLineInBlock = pLineInBlockSave;
         return bFound;
      }
   }

   // Back up from current position within block until we hit 
   // the zero byte in the unused portion of a line header.
   while(*(--pLineInBlock));
   // Now skip to the beginning of this line.
   pLineInBlock -= LN_HDLEN-1;
   bFound = TRUE;

   return bFound;
}

/*--- function nxtlnb ---------------------------------------
*         FIND THE NEXT LINE AFTER A GIVEN LINE
*
*         ENTRY    pBlkIndexEntry CONTAINS THE ADDR OF THE INDEX ENTRY
*                    CORRESPONDING TO THE GIVEN LINE
*                  pLineInBlock CONTAINS THE ADDR OF THE GIVEN LINE
*
*         EXIT     pBlkIndexEntry CONTAINS THE ADDR OF THE INDEX ENTRY
*                    CORRESPONDING TO THE NEXT LINE.
*                  pLineInBlock CONTAINS THE ADDR OF THE NEXT LINE.  It POINTS
*                    TO AN MK.INEND BYTE IF THERE ARE NO MORE TEXT
*                    LINES.
*                  Returns TRUE if we got the next line, else FALSE if 
*                    there are no more lines.
*
*         calls    none
*
*         method   if not at end of block then
*                    skip to next textline
*                    if not at end of block then
*                      RETURN
*                    endif
*                  endif
*                  skip to the next index entry
*                  if at end of index then
*                    RETURN  "end-of-index"
*                  else
*                    RETURN  address of first line in new block
*                  endif
*/
BOOL CScrApp::nxtlnb(Byte8* &pBlkIndexEntry, Byte8 *&pLineInBlock)
{
   BOOL bGotNext=FALSE;
   if(MK_BKEND != *pLineInBlock) {
      // Skip to next line in block.
      int len = pLineInBlock[OF_LNLEN];
      pLineInBlock += len;
      if(MK_BKEND != *pLineInBlock) {
         bGotNext = TRUE;
      }
   }
   if(!bGotNext) {
      // This is the last line in the block, so try the next block.
      if(nxtidx(pBlkIndexEntry)) {
         // We did find the next block.
         Byte8 *pBegOfBlock = blkadr(pBlkIndexEntry);
         pLineInBlock = pBegOfBlock + BLK_HEAD;
         bGotNext = (*pLineInBlock != MK_BKEND);
      }
   }
   return bGotNext;
}

/*--- function nearln --------------------------------------
*         find the index entry addr & textblock addr of a line
*         whose line # is at, after, or before a given line #.
*         (in decreasing order of preference).
*
*         entry    LineNumToFind = addr of a line #
*                  mode = desired program mode
*
*         exit     pBlkIndexEntry = addr of index entry
*                  pLineInBlock = addr of a line within a textblock
*                  Returns false if no line could be found.  (this means
*                    that there are no lines of that mode in memory.)
*
*         CALLS    FINDLN  TO FIND A LINE AT OR AFTER THE LINE #
*                  PRVLNB  TO BACK UP ONE LINE.
*                  nxtidx  to go to next index entry
*
*         uses     all
*
*         method   if no textblocks for this mode then
*                    return
*                  endif
*                  CALL FINDLN
*                  IF not end of block then
*                    return
*                  else
*                    skip to next block
*                    if no next block then
*                      try previous line in old block
*                    endif
*                  endif
*/
BOOL CScrApp::nearln(Byte8* LineNumToFind, TypProgMode mode, 
            Byte8* &pBlkIndexEntry, Byte8 *&pLineInBlock)
{
   BOOL bFound=FALSE;
   int nBlocksInMode = (MD_MSUE==mode) ? ct_bked : ct_bktr;
   if(nBlocksInMode > 0) {
      findln(LineNumToFind, mode, pBlkIndexEntry, pLineInBlock);
      if(MK_BKEND != *pLineInBlock) {
         // Not end of block; return "found".
         bFound = TRUE;
      } else {
         // We hit the end of the block, so try the next block.
         Byte8 *pBlkIndexEntrySave = pBlkIndexEntry;
         if(nxtidx(pBlkIndexEntry)) {
            // We are at next block.  Skip to first line in block.
            pLineInBlock = BLK_HEAD + blkadr(pBlkIndexEntry);
            bFound = TRUE;
         } else {
            // No next block.  Try previous line in old block.
            pBlkIndexEntry = pBlkIndexEntrySave;
            bFound = prvlnb(pBlkIndexEntry, pLineInBlock);
         }
      }
   }
   return bFound;
}

//=====  End of textblock routines  ===============================

/*--- function checke ---------------------
*         CHECK TO SEE IF LINES EXIST IN MEMORY
*
*         ENTRY    'CHECKE' IS CALLED AT THE BEGINNING OF SEVERAL
*                  KEYBOARD-EDITTING ROUTINES.
*                  FG.LNEX  IS TRUE IF THERE IS AT LEAST ONE LINE
*                  IN MEMORY.
*                  FG.RLIN  IS TRUE IF WE'RE RECEIVING LINES
*                  ASYNCHRONOUSLY.
*
*         EXIT     IF THERE ARE LINES IN MEMORY AND WE'RE
*                  NOT BUSY RECEIVING LINES, we return TRUE.
*                  OTHERWISE, WE BEEP AND return FALSE.
*
*         calls    ifbeep  to conditionally beep
*
*         method   if we are in edit mode then
*                    if there are no lines in memory
*                      or lines are arriving from mainframe then
*                      beep
*                      return to caller of caller
*                    else if cursor is on *** beginning of workfile ***
*                      and character<>cursor down and character <> cr
*                      return to caller of caller
*                    else if cursor is on *** end of workfile ***
*                      and character <> cursor up then
*                      return to caller of caller
*                    endif
*                  endif
 */
BOOL CScrApp::checke()
{
	BOOL bOKToEdit=FALSE;
	if(MD_MSUE != fg_mode) {
		bOKToEdit = TRUE;
	} else {
		if(0 == ct_edln) {
			// No lines in memory.
			//!ifbeep();
		} else {
			if(!fg_rlin) {
				// We are not receiving lines from the mainframe.
				if(fg_abow) {
					// Cursor is on the *** beginning of workfile *** line.
					// Is the char one of the few allowed?
					if(m_inchar == '\r' || VK_DOWN == m_inchar) {
						// Yes, it's allowed.
						bOKToEdit = TRUE;
					}
				} else if(fg_oeow) {
					// the cursor is on the *** end of workfile *** line.
					// The only char allowed is up arrow.
					if(VK_UP == m_inchar) {
						// Yes, it's allowed.
						bOKToEdit = TRUE;
					}
				} else {
					// It's on neither beg nor end of workfile, so it's OK.
					bOKToEdit = TRUE;
				}
			}
		}
	}
	return bOKToEdit;
}

//=====  Miscellaneous Terminal mode routines  ====================

/*--- function findtsc -------------------------------------
*         find the line at top of screen on base screen)
*
*         find the line in a textblock that would be at the top
*         of the screen in terminal mode if the very last line
*         in terminal mode were actually displayed on the screen.
*         (this is more useful than it sounds--it is used by
*         routines that do paging back and forth and have to
*         figure out where they are with respect to the clb,
*         the bottom line, etc.)
*
*         entry    none
*
*         exit     pLineInBlock = fwa of the line that would be on the top
*                    of the screen on the base screen.  (within
*                    textblock.)
*                  pBlkIndexEntry = fwa of index entry corresponding to above line.
*
*         CALLS    findlcm to find a line
*                  syserr  to declare a system error
*                  prvlnb  to find the previous line
*
*         method   FIND BOTTOM LINE IN TEXTBLOCK
*                  IF NOT FOUND THEN
*                    SYSTEM ERROR
*                  ENDIF
*                  FOR ISC := SCREEN_SIZE-1 DOWNTO 0 DO
*                    SKIP BACK ONE TEXTBLOCK LINE
*                    IF THERE WAS NO PREVIOUS LINE THEN
*                      BREAK
*                    ENDIF
*                  ENDFOR
 */
void CScrApp::findtsc(Byte8 *&pBlkIndexEntry, Byte8 * &pLineInBlock)
{
   if(!findlcm(ln_bttr, pBlkIndexEntry, pLineInBlock)) {
      CALL_SYSERR("findtsc: current line not found");
   }

   // Skip backwards one screen's worth of lines.
   for(ss_bkln = SC_LNNUM; ss_bkln; ss_bkln--) {
      if(!prvlnb(pBlkIndexEntry, pLineInBlock)) break;
   }

}

/*--- function modlm ---------------------
*         MODIFY THE LEFT MARGIN, IF NEC.)
*
*         ENTRY    HL CONTAINS THE ADDRESS OF A BYTE SPECIFYING
*                    THE LEFT MARGIN.
*                  A  CONTAINS THE CURRENT CURSOR COLUMN NUMBER.
*
*         EXIT     IF THE CURSOR WAS NOT ON THE SCREEN, THE LEFT
*                  MARGIN HAS BEEN CHANGED SO THAT THE CURSOR IS AT
*                  THE LEFT OR RIGHT END OF THE SCREEN LINE.
*                  A  CONTAINS TRUE IF THE LEFT MARGIN HAD TO BE CHANGED
*
*         calls    none
*
*         method   if cursor < left margin then
*                    left margin := cursor
*                    return "true" (* margin changed *)
*                  else if cursor - left margin <= screen size then
*                    return "false" (* margin not changed *)
*                  else
*                    left margin := cursor - screen size - 1 (**Wrong)
*                    return "true" (* margin changed *)
 */
BOOL CScrApp::modlm(int *pLeftMargin, Byte8 cursorPos)
{
	BOOL bChanged=FALSE;
	ScrUtil.DebugOut("modlm: *pLeftMargin=%d, cursorPos=%d", *pLeftMargin, cursorPos);
	if(cursorPos < *pLeftMargin) {
		*pLeftMargin = cursorPos;
		bChanged = TRUE;
	} else if((cursorPos-*pLeftMargin) < SC_LNLEN) {
		bChanged = FALSE;
	} else {
		*pLeftMargin = cursorPos - SC_LNLEN + 1;
		bChanged = TRUE;
	}
	return bChanged;
}

/*--- function chklm ------------------------------------------
*         CHECK LEFT MARGIN and adjust if necessary
*
*         ENTRY    WE'RE IN either program MODE, AND HAVE MOVED THE
*                  cursor.  WE WANT TO CHECK TO MAKE SURE WE
*                  HAVEN'T GONE OFF THE VISIBLE SCREEN.
*
*         EXIT     IF THE CURSOR WAS OFF THE SCREEN, THE LEFT MARGIN
*                  HAS BEEN ADJUSTED and the screen updated.
*
*         calls    dsscrn  to redisplay the screen.
*                  modlm   to modify the left margin.
*                  dstabrl to display the tab ruler in status mode.
*
*         method   call modlm with cursor addr and appropriate margin
*                  if margin was changed by modlm then
*                    redisplay screen for current mode
*                  endif
 */
void CScrApp::chklm()
{
   int *pMargin;
   switch(fg_mode) {
      case MD_MSUE:
         pMargin = &lmargin;
         break;
      case MD_TRM:
         pMargin = &trlmarg;
         break;
      default:
         pMargin = &tabmarg;
   }
   if(modlm(pMargin, clb_col)) {
      // Margin moved, so redisplay screen.
      if(MD_STA == fg_mode) {
         //! write this: dstabrl();
      } else {
         dsscrn();
      }
   }
}

/*--- function trbotm ---------------------------------------
*         put cursor on bottom line in term mode
*
*         if the cursor's not already on the bottom line, put
*         it there and adjust screen so the last line is displayed
*         at the bottom.
*
*         entry    none
*
*         exit     the cursor is on the bottom line and the clb
*                  has the bottom line.
*
*         calls    ibf2clb to move the bot line to the clb
*                  movelno to move a line number
*                  dsscrn  TO DISPLAY THE SCREEN
*                  findtsc to find the top line on screen
*                  chklm   to change left margin if necessary
*
*         METHOD   SET LEFT MARGIN TO ZERO
*                  IF CURSOR NOT ON BOTTOM LINE THEN
*                    put bottom line in clb
*                    find the appropriate top line
*                    set cursor_on_bottom_line := true
*                  endif
*                  modify left margin if cursor would be off screen
*                  REDISPLAY THE SCREEN
 */
void CScrApp::trbotm()
{
   trlmarg = 0;   // left margin to 0
   if(!fg_cobl) {
      // Cursor is not already on bottom line
      ibf2clb();
      Byte8 *pBlkIndexEntry, *pLineInBlock;
      findtsc(pBlkIndexEntry, pLineInBlock);  // find top line for bottom screen
      movelno(pLineInBlock, ds_tpln);  // Make it the top line.
      fg_cobl = TRUE;
   }
   chklm(); // check left margin
   dsscrn();
}

/*--- function ibf2clb -----------------------------
*         copy the terminal mode bottom line buffer to the clb,
*         along with associated variables.
*
*         entry    none
*
*         exit     none
*
*         calls    copybuf to copy a buffer
*                  movelno to copy a line number
*
*         method   simple
*/
void CScrApp::ibf2clb()
{
   copybuf(bf_inln, clb_buf);
   movelno(ln_bttr, clb_lno);
   clb_col = pt_inln;   // copy cursor pos
   clb_fkc = ct_bfkc;   // copy ordinal of first keyboard char
   clb_alt = FALSE;     // clear 'altered' flag
   clb_scl = SC_LNNUM;  // screen line # of clb
}

/*--- function int2clb -----------------------------------
*         COPY INTERNAL LINE TO CLB
*
*         ENTRY    pLineInBlock POINTS TO AN INTERNALLY-FORMATTED LINE.
*
*         EXIT     THE LINE HAS BEEN COPIED TO THE CURRENT LINE
*                  BUFFER, IN STRAIGHT ASCII.
*                  CLB.LNO CONTAINS THE LINE'S LINE #.
*                  THE LINE # HAS BEEN DISPLAYED ON THE STATUS LINE.
*
*         calls    lno2asc to display the current line number
*                  movei   to move a string of bytes
*                  ds1line to unpack a packed line
*
*         method   if in edit mode then
*                    display line number on status line
*                  endif
*                  move line header to clb line header
*                  call ds1line to unpack line
 */
void CScrApp::int2clb(Byte8 *pLineInBlock)
{
   if(MD_MSUE == fg_mode) {
      //!
   }
   // copy the line information in the line header to the
   // clb variables.  rely on the fact that the line variables
   // are set up in memory the same way the line header is.
   movei(pLineInBlock, clb_lno, LN_HDLEN);

   // Unpack line to current line buffer.
   ds1line(pLineInBlock, clb_buf, LN_TXLEN, 0);
}

/*--- function dstrclb -------------------------------
*         display the current line buffer for terminal mode.
*         the cursor may or may not be on the bottom line.
*
*         entry    trlmarg = left margin
*                  ds.trcl = address of current line.
*
*         exit     none
*
*         calls    movei   to move bytes
*                  dislnor to display the line (ibm)
*
*         method   copy buffer to screen starting at left margin
*                  set horz cursor position
 */
void CScrApp::dstrclb()
{
   // Copy current line buffer to appropriate line of screen.
   movei(clb_buf+trlmarg, ds_trcl, SC_LNLEN);
   // Set cursor pos
   dp_chor = clb_col - trlmarg;
   dp_cvrt = clb_scl;
	CauseRepaint();	//! may want to optimize this.
}

/*--- function dsclb ---------------------
 *  Display the current line buffer, for editor or terminal mode.
 */
void CScrApp::dsclb()
{
	if(MD_MSUE==fg_mode) {
		//! dsedclb();
	} else {
		dstrclb();
	}
}

/*--- function dstribf ----------------------------
*         display the bottom line in terminal mode.
*         Call this if the cursor is not on the bottom line.  
*        (If it is, call dstrclb instead.)
*
*         entry    botline is the address of the bottom line on screen
*                  fg.blos is true iff bottom line is displayable
*                  trlmarg is left margin
*                  bf.inln is the bottom line.
*
*         exit     for ibm version, fg.udbl is true (to display
*                    bottom line in isrvrt) if the bottom line is
*                    actually on the screen.
 */
void CScrApp::dstribf()
{
   // Is the bottom line displayable on the screen?
   if(fg_blos) {
      movei(bf_inln+trlmarg, botline, SC_LNLEN);
   }
}

/*--- function ds1line ------------------------------------
*         DISPLAY 1 LINE FROM TEXTBLOCK UPON SCREEN
*
*         ENTRY    pLineInBlock CONTAINS THE ADDR OF THE LINE (WITHIN A
*                    TEXTBLOCK) TO BE DISPLAYED.
*                  pScreenLn CONTAINS THE FWA OF THE PHYSICAL SCREEN LINE TO
*                    BE DISPLAYED.
*                  nbytes_to_display  CONTAINS THE # OF BYTES TO UNPACK (80 FOR DISPLAY
*                    OF SCREEN LINE, 140 FOR FULL UNPACKED LINE).
*                  chars_to_skip = # OF CHARS IN THE INTERNAL LINE TO SKIP BEFORE
*                    STARTING TO DISPLAY THEM.  USUALLY EITHER 0 OR
*                    LMARGIN (WHICH ITSELF IS USUALLY ZERO).
*                  fg.fdb is true if we should use fast display blanking
*                    else false
*
*         exit     the specified line has been unpacked into the
*                    designated area.  if "fg.fdb" is true, it has
*                    been terminated with an ontel "logical end of
*                    line" byte, else it has been padded out with
*                    blanks.
*
*         calls    fill    to fill out a line
*
*         method   (*  Skip to left margin position in text line.
*                      Copy remaining characters to screen, until
*                      end-of-line or end-of-screen.  Bytes with
*                      bit 7 set signify a sequence of packed
*                      characters, which need to be unpacked.
*                      At end, blank screen line from current
*                      position to end of line, if necessary.  *)
*
*                  skip to beginning of packed text in input line
*                  (* skip past chars up to left margin *)
*                    while chars_to_skip > 0 do
*                      skip chars by incrementing line_ptr as below
*                    endwhile
*                  (* copy the remaining characters to screen *)
*                    if chars_left_on_screen = 0 then return
*                    increment packed_bytes_left
*                    repeat
*                    copyln:
*                      decrement packed_bytes_left
*                      if packed_bytes_left = 0 then goto endcopy
*                      char := line_ptr^
*                      increment line_ptr
*                      if char is tagged then goto dupstart
*                      screen_ptr^ := char
*                      increment screen_ptr^
*                      decrement chars_left_on_screen
*                    until chars_left_on_screen = 0
*                    return
*                  (* process a sequence of duplicated characters *)
*                  dupstart:
*                    dupcount := line_ptr^
*                    repeat
*                      screen_ptr^ := char
*                      increment screen_ptr^
*                      decrement chars_left_on_screen
*                      if chars_left_on_screen = 0 then return
*                      decrement dupcount
*                    until dupcount = 0
*                    increment line_ptr
*                    decrement packed_bytes_left
*                    if packed_bytes_left <> 0 then goto copyln
*                  (* fill out the end of the screen line *)
*                  endcopy:
*                    if fast_display_blanking then
*                      screen_ptr^ := logical end of line
*                    else
*                      fill(screen_ptr,' ',chars_left_on_screen)
*                    endif
 */
void CScrApp::ds1line(Byte8 *pLineInBlock, Byte8 *pScreenLn, 
           int nbytes_to_display, int chars_to_skip)
{
   Byte8 n_packed_chars = pLineInBlock[OF_LNLEN] - LN_HDLEN;
   Byte8 *pPacked = pLineInBlock+LN_HDLEN;
   Byte8 ndups=0, mybyte;
   BOOL bInDupSeq=FALSE;

   while(n_packed_chars && nbytes_to_display) {
      if(bInDupSeq) {
         if(0 == --ndups) {
            bInDupSeq = FALSE;
            n_packed_chars--;
            if(!n_packed_chars) break;
         }
      }
      if(!bInDupSeq) {
         mybyte = *(pPacked++);
         n_packed_chars--;
         if(0x80 & mybyte) {
            mybyte &= 0x7f;
            ndups = *(pPacked++);
            bInDupSeq = TRUE;
         }
      }
      if(chars_to_skip) {
         chars_to_skip--;
      } else {
         *(pScreenLn++) = mybyte;
         nbytes_to_display--;
      }
   }
   
   // Blank-fill the line.
   while(nbytes_to_display--) *(pScreenLn++) = ' ';
}

/*--- function dstrsc --------------------------------
*         display the entire screen in terminal mode.
*
*         entry    ds.tpln contains the line number to start
*                    displaying at.
*                  fg.ltsc is true if there is less than an
*                    entire screen's worth of lines.
*                  ct.trln contains the number of lines in
*                    terminal mode.
*                  fg.fdb is true if we should use the ontel's
*                    "fast dynamic blanking".
*                  current line buffer information is in the clb
*                    variables, as always.
*
*         exit     ds.trcl has the addr of the screen line for clb
*                  clb.scl has the screen line # of the clb
*
*         calls    fill    to blank a line
*                  dstrclb to display the clb
*                  dstribf to display the bottom line
*                  nxtlnb  to skip to the next line
*                  complno to compare two line #'s.
*
*         method   (*  Display some blank lines at top if necessary.
*                      Loop through lines to display, displaying
*                      them from a textblock, the CLB, or the
*                      bottom line buffer, as appropriate. *)
*
*                  LINES_TO_DISPLAY := MIN(SCREEN_SIZE,TOTAL_LINES)
*                  if lines_to_display < screen_size then
*                    blank screen_size - lines_to_display lines
*                  endif
*                  set screen_pointer to first screen line to display
*                  for disline := lines_to_display downto 1 do
*                    if current line is in clb then
*                      display line from clb
*                    else if current line is in bottom_line_buf then
*                      display line from bottom_line_buf
*                    else
*                      display line from textblock
*                    endif
*                    advance screen_pointer to next line
*                    move to next line in textblock
*                  endfor
*/
void CScrApp::dstrsc()
{
   fg_blos = FALSE;
   int ss_dsnl = SC_PHSCN;
   Byte8 *pNextScreenLn = screen;
   if(fg_ltsc) {
      // we do not have enough lines to fill the screen, so we need
      // to blank the first few lines on the screen.
      int lines_to_blank = SC_PHSCN-ct_trln;
      fill(screen, SC_LNLEN*lines_to_blank, ' ');
      pNextScreenLn += SC_LNLEN*lines_to_blank;
      ss_dsnl = ct_trln;
   }
   // the top part of the screen has been blanked if necessary.
   // now set things up to start displaying text.
   //Byte8 *pBlkIndexEntry, *pLineInBlock;
   findlcm(ds_tpln, ds_crea, ds_crla);
   do {
      //ScrUtil.DebugOut("ds_crla=%s clb_lno=%s", lno2StdString(ds_crla).c_str(), lno2StdString(clb_lno).c_str());
      // if this line is in the clb, display it from the clb.
      if(0==complno(clb_lno, ds_crla)) {
         // Compute the screen line # 
         clb_scl = SC_PHSCN - ss_dsnl;
         ds_trcl = pNextScreenLn;
         dstrclb();
      } else if(0==complno(ln_bttr,ds_crla)) {
         // This is the line at the bottom of the screen.
         // Display it from the bottom line buffer.
         fg_blos = TRUE;   // bottom *is* displayable on screen.
         dstribf();        // display from bottom line buffer
      } else {
         // the line does not have a copy in the clb or bottom line
         // buffer, so it must be displayed from the textblock.
         ds1line(ds_crla, pNextScreenLn, SC_LNLEN, trlmarg);
      }
      pNextScreenLn += SC_LNLEN;
      nxtlnb(ds_crea, ds_crla);
   } while(--ss_dsnl);
}

/*--- function dsscrn -----------------------------------
*         display the edit mode or terminal mode screen, depending
*         on the current program mode.
*
*         entry    ds.tpln contains the line number to display at
*                    the top of the screen.
*                  fg.mode contains the current mode.
*                  fg.dnds is true if we should not display the
*                    screen, after all.
*                  fg.colr indicates whether the color adapter is
*                    being used.
*
*         exit     If fg.dnds is false, the screen has been displayed,
*                    and ct.nlnd has been cleared.
*                  if it was true, the screen has not been changed.
*
*         calls    dsedsc  to display edit mode screen
*                  dstrsc  to display term mode screen
*                  dsstsc  to display status mode screen
*                  dshlsc  to display help screen (ibm)
*                  updscn  to copy the screen to video ram (ibm)
*                  dsv2sc  to display the vt200 screen.
*
*         method   if not "do not display screen" then
*                    if in vt200 mode,
*                      display screen for vt200 mode
*                    endif
*                    if mode = edit mode's command mode then
*                      mode = edit
*                    endif
*                    if mode = edit then
*                      if behind in processing input then
*                        set "do not display screen" flag
*                      else
*                        display screen for edit mode
*                      endif
*                    else if mode = terminal then
*                      display screen for terminal mode
*                    else (* status mode *)
*                      display screen for status mode
*                    endif
*                    restore correct mode
*                  endif  (* not "do not display" *)
*/
void CScrApp::dsscrn()
{
   if(!fg_dnds) {
      //! flesh out edit mode, vt200, etc.
      // check to make sure that the top of the screen is still in memory.
      Byte8 *pBlkIndexEntry, *pLineInBlock;
      if(nearln(ds_tpln, fg_mode, pBlkIndexEntry, pLineInBlock)) {
         // 
         movelno(pLineInBlock, ds_tpln);
      }

      //! check to see if we are behind in processing input.


      if(MD_TRM == fg_mode) {
         dstrsc();
      }
   }
   CauseRepaint();
}

void CScrApp::litekey()
{
	CauseRepaint();
}

void CScrApp::blnkscr()
{
   fill(screen, SC_LNLEN*SC_LNNUM+SC_LNLEN, ' ');
}

void CScrApp::escape()
{
}

void CScrApp::v2prec()
{

}

/*--- function getcrch --------------------------------------
*         get the char at the cursor in the current line
*
*         entry    clb.buf holds the current line.
*                  clb.col is the index into clb.buf for the current
*                    character.
*
*         exit     Returns = the current character.
*                  e = the cursor position (CLB.COL)
*                  hl = address of the current character.
 */
Byte8 CScrApp::getcrch(Byte8 *&pCurChar)
{
   pCurChar = clb_buf + clb_col;
   return clb_buf[clb_col];
}

/*--- function chinclb --------------------------------------
*         place a character in the current line buffer at the cursor,
*         paying attention to the 'insert' flag.
*
*         entry    inchar = the character
*
*         exit     None
*
*         calls    getcrch to get position of current character
*
*         method   if in insert mode then
*                    move bytes from end of clb to cursor back one
*                  endif
*                  place the char at the cursor
 */
void CScrApp::chinclb(Byte8 inchar)
{
   if(fg_insr) {
      // We are in insert mode.  Move chars down, starting at cursor.
      moveq(clb_buf+clb_col, clb_buf+LN_TXLEN-2, 1);
   }
   Byte8 *pdest;
   getcrch(pdest);
   *pdest = inchar;
}

/*--- function incebf ------------------------------------
*         increment a cursor location unless that action would
*         take the cursor beyond the standard length of a buffer.
*
*         entry    pCursorPos = address of a cursor byte
*
*         exit     *pCursorPos may have been incremented.
*                  Returns 1 if cursor was actually incremented.
 */
BOOL CScrApp::incebf(Byte8 *pCursorPos)
{
   Byte8 cursor = *pCursorPos;
   BOOL bInc = FALSE;
   if(cursor < LN_TXLEN-1) {
      (*pCursorPos)++;
      bInc = TRUE;
   }
   return bInc;
}

/*--- function tbarriv ---------------------
*         process cursor arriving at new column
*
*         figure out what to do when the cursor arrives at a new
*         column.  this, of course, depends on the current tabset.
*         usually, there's nothing to do, but if we wind up at a
*         'd' or 's' tab stop, for example, some processing is
*         required.
*         this routine is called by all routines that move the cursor.
*
*         entry    clb.col contains the new cursor location.
*
*         exit     none
*
*         calls    chklm   to adjust the left margin if nec.
*                  DSCLB   TO DISPLAY THE CURRENT LINE.
*                  tbstate to process the current state of the tab fsa
*                  ifbeep  to make a beep
*
*         METHOD   if in edit mode then
*                    STATE := NULL
*                    REPEAT
*                      CASE TABSET[COL] OF
*                        D:  STATE := DUP
*                        S:  STATE := SKIP
*                        B:  BEEP
*                            STATE := NULL
*                        R:
*                            STATE := NULL
*                        -:  (NOTHING)
*                        OTHERS:  STATE := NULL
*                      ENDCASE
*                      CALL TBSTATE(STATE)  (* PROCESS STATE *)
*                    UNTIL  STATE = NULL
*                  endif
*                  redisplay current line
*
*         note:    be careful in this routine to avoid recursion.
*                  recursion might be an attractive way to implement
*                  some tab types, but there isn't enough stack space
*                  to handle a very long tab field this way. */
void CScrApp::tbarriv()
{
	//! need to flesh this out 
	chklm();
	dsclb();
}

/*--- function endrec -------------------------
*         END & TRANSMIT A RECORD TO ASYNCH PORT)
*
*         ENTRY    A BUFFER OF CHARACTERS TO BE SENT TO THE MAINFRAME
*                  HAS BEEN ASSEMBLED.  WE'RE READY TO TRANSMIT THEM.
*                  CURRENTLY, WHEN ENDREC IS CALLED, THE COMPLETE
*                  RECORD HAS BEEN TRANSMITTED, EXCEPT FOR THE
*                  CARRIAGE RETURN AT THE END.  IF A MORE SOPHISTICATED
*                  COMMUNICATIONS PROTOCOL IS EVER ADOPTED, THIS
*                  ROUTINE WILL CHANGE.
*
*         EXIT     THE RECORD HAS BEEN ENCODED & TRANSMITTED.
*                    CURRENTLY, THIS JUST MEANS THAT A CR IS SENT.
*                  The line feed count has been decremented.
*                  The timer to count down how long ct.lf is
*                    negative has been reset.
*
*         calls    putasy  to transmit a character */
void CScrApp::endrec()
{
	putasy('\r');
	ct_lf--;
}

//=====  MSU EDITOR mode  =========================================
// Some MSU EDITOR mode routines are also used by other modes.

//--- Chars received from mainframe.

//--- Chars received from keyboard.

/*--- function kyprint -------------------------------------
*         PROCESS A PRINTABLE CHAR ENTERED FROM KEYBOARD
*
*         ENTRY    A PRINTABLE CHAR HAS BEEN TYPED IN.
*                  'A' CONTAINS THAT CHAR.
*
*         EXIT     THE CHARACTER HAS BEEN ENTERED INTO THE CLB
*                  AND THE LINE HAS BEEN REDISPLAYED.
*                  clb.alt is true.
*
*         calls    checke  to make sure it's ok to do anything
*                  getcrch to get the character at the cursor
*                  chinclb to insert a character into the clb
*                  incebf  to increment a byte
*                  tbarriv to process a new tab column
*                  ifbeep  to issue a beep or a click
*
*         method   make sure it's ok to do this
*                  if in verify mode then
*                    if input char <> character at cursor then
*                      beep
*                      return
*                    endif
*                  else
*                    call chinclb to do the real work
*                    set "line altered" flag
*                  endif
*                  advance the cursor
*                  process new tab column
 */
void CScrApp::kyprint(Byte8 inbyte)
{
	if(!checke()) return;
	chinclb(inbyte);
	clb_alt = TRUE;
	incebf(&clb_col);
	tbarriv();
}

/*--- function cursrup ---------------------------------
*         MOVE THE CURSOR UP ONE LINE
*
*         Move the cursor up one line in terminal or edit mode.
*
*         ENTRY    THE USER HAS HIT THE UP-ARROW KEY
*
*         EXIT     THE CURSOR HAS MOVED UP ONE LINE. IF NECESSARY,
*                  THE SCREEN HAS BEEN SCROLLED UP TO ACCOMODATE
*                  THE CURSOR.  COPIES OF THE LINE IN THE CLB HAVE
*                  BEEN UPDATED IF NECESSARY.
*
*         calls    checke  to make sure it's ok to move cursor
*                  ifalt   to update copies of clb if altered
*                  complno to compare line numbers
*                  clb2ibf to move the clb to the bottom line buffer
*                  findlcm to find a line in the current mode
*                  prvlnb  to skip back one line
*                  yeswait to set 'waiting for mainframe'
*                  putstr  to output a string of bytes to the mainframe
*                  xmtlno  to output an ascii editor line number
*                  int2clb to unpack a textblock line to the clb
*                  movelno to move a line number
*                  dsscrn  to redisplay the screen
*
*         method   (*  This routine is used for both edit and terminal
*                      modes, so we do some special checking at the
*                      beginning for when we are at the beginning or
*                      end of textblock memory.
*                      Then, we find the current line and try to back
*                      up.
*                      In edit mode, if we can't find a previous
*                        line, we have to go through the painful process
*                        of figuring out whether to send a LIST command.
*                      Otherwise, figure out whether the screen needs to
*                        be scrolled.  *)
*
*                  check to make sure it's ok to move cursor
*                  if in edit mode then
*                    clear "can't display ***bow*** line" flag
*                    update current line if altered
*                    if ***bow*** on screen and on first line then
*                      set "cursor on ***bow*** line" flag
*                      goto display
*                    else if on ***eow*** line then
*                      clear "cursor on ***eow*** line" flag
*                      goto display
*                    endif
*                  else  (* terminal mode *)
*                    if cursor is on bottom line then
*                      if there's only one line in terminal mode then
*                        return
*                      else
*                        clear "cursor on bottom line" flag
*                      endif
*                    endif
*                  endif
*                  find current line and try to back up one line
*                  if none found then
*                    if in terminal mode then return
*                    else
*                      if we have first ewfile line in memory then
*                        return
*                      else
*                        set flags to check for beg of workfile
*                        send 'list' command to mainframe
*                        return
*                      endif
*                    endif
*                  else  (* previous line found *)
*                    put previous line in clb
*                    if at top of screen then
*                      make this new line the top of screen
*                    endif
*                    display:
*                    redisplay the screen
*                  endif
 */
void CScrApp::cursrup()
{
   //! checke();
   if(MD_TRM != fg_mode) {
      //! flesh out
   } else {
      // Terminal mode
      if(fg_cobl) {
         // if there's only one line in terminal mode, just return.
         if(ct_trln <= 1) return;
         fg_cobl = FALSE; // cursor no longer on bot line
         clb2ibf();
      }
   }

   if(true) { //! not sure about this
      // NOW, FIND THE LINE IN A TEXTBLOCK & TRY TO BACK UP.
      Byte8 *pBlkIndexEntry, *pLineInBlock;
      findlcm(clb_lno, pBlkIndexEntry, pLineInBlock);
      if(!prvlnb(pBlkIndexEntry, pLineInBlock)) {
         // There is no previous line.
         if(MD_TRM == fg_mode) {
            return;
         } else {
            // Edit-mode only processing.
            //! flesh out
         }
      } else {
         // PREVIOUS LINE IS IN MEMORY, SO MAKE THAT ONE THE CURRENT LINE.
         int2clb(pLineInBlock);
         // IF WE'RE AT THE TOP OF THE SCREEN ALREADY, WE'D BETTER
         // MAKE THIS PREV LINE THE TOP LINE.
         if(0 == clb_scl) {
            // Prev line was at the top, so make this new line the top.
            movelno(clb_lno, ds_tpln);
         }
         dsscrn();
      }
   }
}

/*--- function cursrdn ---------------------------------
*         MOVE THE CURSOR DOWN ONE LINE
*
*         ENTRY    THE USER HAS JUST PRESSED THE CURSOR DOWN KEY
*                  IN MSU EDITOR MODE.
*
*         EXIT     THE CURSOR HAS MOVED DOWN ONE LINE.  IF NEC.,,
*                  - THE CURRENT LINE HAS BEEN TRANSMITTED
*                  - THE SCREEN HAS BEEN ROLLED UP ONE LINE
*                  - A LINE HAS BEEN REQUESTED OF MSU EDITOR
*
*         calls    checke  to check whether we're allowed to do this
*                  ifalt   to update copies of cld if altered
*                  findlcm to find a line in the current mode
*                  nxtlnb  to find the next line
*                  askline to ask the mainframe for more lines
*                  int2clb to unpack a line into the clb
*                  movelno to move a line number
*                  dsscrn  to redisplay the screen
*
*         method   check to make sure it's ok to move cursor.
*                  update current line if it's been altered
*                  if we are on the *** beg of workfile *** line then
*                    clear this flag
*                    goto display
*                  else
*                    find the current line in textblock memory
*                    skip to next line
*                    if there's no next line then
*                      if we have the last line of the ewfile then
*                        return
*                      else
*                        set flags for end of workfile checking
*                        send 'list' command to mainframe
*                      endif
*                    else
*                      move next line into current line buffer
*                    endif
*                    if we are on the bottom line of screen then
*                      set 'do not display ***bow***' flag
*                      if '*** beg of workfile ***' is not displayed
*                        move top-of-screen pointer down one line
*                      endif
*                    endif
*                    display:
*                    redisplay the screen
*                  endif
 */
void CScrApp::cursrdn()
{
}

/*--- function kydelch ---------------------
*         PROCESS THE 'DELETE CHARACTER' KEY
*         This is used for both terminal mode and EDITOR mode.
*
*         ENTRY    THE USER HAS JUST HIT THE 'DELETE CHARACTER' KEY
*
*         EXIT     THE CHARACTER AT THE CURSOR HAS BEEN DELETED AND
*                  THE REST OF THE LINE TO THE RIGHT HAS BEEN SHIFTED
*                  ONE COLUMN TO THE LEFT TO FILL THE DELETED CHAR'S
*                  PLACE.
*
*         calls    checke  to make sure it's ok to do this
*                  dsclb   to display the clb
*
*         method   make sure it's ok to do this
*                  mark the line as altered
*                  nbytes := size of buffer - cursor pos - 1
*                  charidx := cursor position
*                  while nbytes <> 0 do
*                    clb[charidx-1] := clb[charidx]
*                    increment charidx
*                    decrement nbytes
*                  endwhile
*                  put blank at end of clb
*                  if there's system prompt info on the line then
*                    if cursor pos. < first user character then
*                      decrement first user character position
*                    endif
*                  endif
 */
void CScrApp::kydelch()
{
	if(checke()) {
		clb_alt = TRUE;
		if(clb_col < LN_TXLEN-1) {
			moveq(clb_buf+clb_col+1, clb_buf+LN_TXLEN, -1);
		}
		clb_buf[LN_TXLEN-1] = ' ';
		// if there's a system prompt on this line, then
		// if the character being deleted is a system prompt char,
		// decrement the "first user character position" byte.
		if(MK_NOKYC != clb_fkc) {
			if(clb_col < clb_fkc) {
				clb_fkc--;
			}
		}
		dsclb();
	}
}

/*--- function kyinsrt ---------------------
*         TOGGLE REPLACE/INSERT MODE)
*
*         ENTRY    THE REPLACE/INSERT MODE TOGGLE KEY HAS JUST
*                  BEEN PRESSED.
*
*         EXIT     THE REPLACE/INSERT MODE FLAGS HAVE BEEN TOGGLED
*                  AND THE INSERT MODE LIGHT HAS BEEN SET ACCORDINGLY
*
*         calls    litekey to light up the lightable keys
 */
void CScrApp::kyinsrt()
{
	fg_insr = !fg_insr;
	litekey();
}


//=====  Terminal Mode  ===========================================
//  Chars received from mainframe

/*--- function trrchar -----------------------------------
*         process a printing character (also some control chars)
*         received from the mainframe in terminal mode.
*
*         entry    m_inchar = the char
*
*         exit     none
*
*         calls    chklm   to adjust left margin if necessary
*                  dstrclb to display the clb
*                  dstribf to display the bottom line
*                  chinclb to add a character to the clb
*                  incebf  to move the cursor
*
*         method   (*  Add the incoming character to the CLB, or
*                      to the bottom line buffer if appropriate.
*                      If this character was echoed from the
*                      keyboard, adjust the "first keyboard echoed"
*                      field and the left margin, if necessary.
*                      Otherwise, throw the character on the
*                      screen as quickly as possible and exit.  *)
*
*                  if cursor is on bottom line then
*                    add char to clb at cursor
*                    if char was echoed from keyboard then
*                      decrement count of echoed chars
*                      if cursor pos < first keyboard char pos then
*                        first keyboard char pos := cursor pos
*                      endif
*                      if cursor not at end of buffer then
*                        increment cursor pos.
*                      endif
*                      adjust left margin if necessary
*                      display clb
*                    else  (* not an echoed char *)
*                      if cursor not at end of line then
*                        put character directly on screen
*                        increment cursor position.
*                      endif
*                    endif
*                  else  (* cursor is not on bottom line *)
*                    add char to bottom line buffer
*                    if char pointer not at end of line then
*                      increment char pointer
*                    endif
*                    display bottom line
*                  endif
 */
void CScrApp::trrchar()
{
   //ScrUtil.DebugOut("trrchar char=%c (0x%2.2x)", m_inchar, m_inchar);
	if(fg_cobl) {
		// cursor is on bottom line
      chinclb(m_inchar);
		// if char was echoed from keyboard, then do related bookkeeping.
		// also, if the char was echoed, the user is typing and
		// characters are arriving slowly enough (most likely) for
		// us to adjust the left margin when needed.
		if(ct_cstm) {
			ct_cstm--;	// # of chars we expect to have echoed.
			if(clb_fkc > clb_col) {
				clb_fkc = clb_col;
			}
			incebf(&clb_col);
			chklm();
			dstrclb();
		} else {
			// Char was not echoed from the keyboard.
			incebf(&clb_col);

			// to speed things up, for chars from the mainframe on the
			// bottom line that are not echoed--that is to say, most of the
			// time, we just put the char right on the screen.
			if(dp_chor < SC_LNLEN-1) {
				botline[dp_chor] = m_inchar;
				dp_chor++;
			}
		}
   } else {
		// the cursor was not on the bottom line, so the newly-arrived
		// character should go in the bottom-line buffer.
      bf_inln[pt_inln] = m_inchar;
      if(pt_inln < LN_TXLEN-1) pt_inln++;
		dstribf();
   }
}

/*--- function trrtab ---------------------
 *  Handle a tab received in terminal mode, by expanding it to spaces.
 *  This was not a part of the original SCREDIT.
 */
void CScrApp::trrtab()
{
	// Determine current cursor column based on whether we are on bot line.
	Byte8 curcol = clb_col;
	if(!fg_cobl) curcol = pt_inln;
	Byte8 nspaces = (8*((8+curcol) / 8) ) - curcol;
	while(nspaces--) {
		m_inchar = (Byte8)' ';
		trrchar();
	}
}

/*--- function trrcr ----------------------------------
*         process a carriage return from the mainframe in term mode.
*
*         entry    fg.colb says whether cursor is on bottom line.
*
*         exit     The virtual and physical cursors have been
*                    sent to the left margin.
*
*         calls    dsscrn  to display the entire screen
*                  dstrclb to display the clb
*                  dstribf to display the bottom line
*
*         method   echoed_chars_expected := 0
*                  if cursor is on bottom line then
*                    cursor_pos := 0
*                    if left_margin <> 0 then
*                      left_margin := 0
*                      display screen
*                    endif
*                    display clb
*                  else  (* cursor not on bottom line *)
*                    buffer_pointer := 0
*                    if not behind in processing characters then
*                      display bottom line
*                    endif
*                  endif
 */
void CScrApp::trrcr()
{
   ScrUtil.DebugOut("trrcr: fg_cobl=%d fg_dnds=%d", fg_cobl, fg_dnds);
   ct_cstm = 0;   // set # of echoed chars expected from mainframe
   dp_chor = 0;   // cursor to column 1
   if(fg_cobl) {
      // Cursor is on bottom line
      clb_col = 0;
      if(trlmarg > 0) {
         trlmarg = 0;
         dsscrn();
      }
      // display clb if not behind
      if(!fg_dnds) {
         dstrclb();
      }
   } else {
      // the cursor is not on the bottom line.
      pt_inln = 0;
      dstribf();
   }
}

/*--- function trrlf --------------------------------------
*         process a line feed received from the mainframe in
*         terminal mode.
*
*         entry    fg.dnds is true if we shouldn't display screen;
*                    this applies only for 8086's.
*
*         exit     none
*
*         calls    addlbtm to add the line to bottom of terminal mode
*                  dsscrn  to display the screen
*                  complno to compare line numbers
*                  movelno to move a line number
*                  movei   to move most of a screen image
*                  updscn  to update the video ram (ibm)
*                  dstrclb to display the current line
*
*         method   call addlbtm to add line to textblocks
*                  increment number of suppressed lines
*                  if number of suppressed lines >= force display count
*                    'do not display flag' = false
*                  endif
*                  if the cursor is on the bottom line then
*                    just copy the screen image up by one line
*                  else
*                    redisplay the entire screen
*                  endif
*/
void CScrApp::trrlf()
{
   ScrUtil.DebugOut("trrlf called");
   addlbtm();
   //! add conditional screen redisplay logic here.
   dsscrn();
}

//----- Terminal mode:  Input from keyboard  ------------------------

/*--- function trkcudn --------------------------------------
*         process the 'move cursor down' key in terminal mode.
*
*         entry    none
*
*         exit     none
*
*         calls    findlcm to find a line
*                  nxtlnb  to find the next line
*                  complno to compare line numbers
*                  trbotm  to send the cursor to the bottom of screen
*                  int2clb to unpack a line into the clb
*                  dsscrn  to display the screen
*                  movelno to move a line number
*
*         method   if cursor not on bottom line then
*                    find the line after the clb
*                    if it's the last line in memory then
*                      call trbotm to send cursor to bot line
*                    else
*                      unpack the line into the clb
*                      if the cursor is on last line of screen then
*                        move the 'top of screen' pointer down one line
*                      endif
*                      redisplay the entire screen
*                    endif
*                  endif
 */
void CScrApp::trkcudn()
{
   if(!fg_cobl) {
      // Advance pointer to current line by one line.
      Byte8 *pBlkIndexEntry, *pLineInBlock;
      findlcm(clb_lno, pBlkIndexEntry, pLineInBlock);
      nxtlnb(pBlkIndexEntry, pLineInBlock);
      if(0==complno(ln_bttr, pLineInBlock)) {
         // New line is the bottom line, so send the cursor to this bot line.
         trbotm();
      } else {
         int2clb(pLineInBlock);
         if(SC_LNNUM == clb_scl) {
            // Cursor is on bottom line of screen, so move 
            // "top of screen" pointer down one line.
            findlcm(ds_tpln, pBlkIndexEntry, pLineInBlock);
            nxtlnb(pBlkIndexEntry, pLineInBlock);
            movelno(pLineInBlock, ds_tpln);
         }
         dsscrn();
      }
   }
}

/*--- function trklft ----------------------------------
*         process the left arrow or backspace in terminal mode.
*
*         ENTRY    a = the left arrow key or backspace.
*
*         exit     none
*
*         calls    putasy  to send a character
*                  decbnz  to decrement a byte
*                  dstrclb to display the clb
*                  chklm   to check the left margin
*                  ifscrol to check scroll lock (ibm)
*                  kywinlf to window left if scrolling (ibm)
*
*         method   if ibmpc and scroll lock, just call scrolling routine
*                  if cursor is on bottom line
*                    send a bs
*                    if character was backspace (not left arrow) then
*                      increment count of destructive backspaces
*                    endif
*                  else
*                    move cursor left if not all the way to the left
*                    check the left margin
*                    display the line
*                  endif
 */
void CScrApp::trklft()
{
   if(fg_cobl) {
      putasy('\b'); // send char to mainframe
		// if the character typed was a backspace, increment the
		// count of "deletable" backspaces sent.  we do this to
		// keep track of the difference between a left arrow and
		// a backspace.
		if('\b' == m_inchar) {
			ct_debs++;	// increment count of backspaces
		}
   } else {
		decbnz(&clb_col);
		chklm();
		dstrclb();
	}
}

/*--- function trkrgt ---------------------
*         process the 'cursor right' key from the keyboard in
*         terminal mode.
*
*         entry    none
*
*         exit     none
*
*         calls    putasy  to send a character
*                  chklm   to check the left margin
*                  dstrclb to display the clb.
*                  ifscrol to check scroll lock (ibm)
*                  kywinrt to window right (ibm)
*
*         method   if ibmcp and scroll lock is on, just call scroll rtn.
*                  if cursor is on bottom line
*                    send the character at the cursor
*                    increment the count of characters sent
*                  else  (* not on bottom line*)
*                    if cursor not at right margin
*                      move cursor to right
*                      check left margin
*                      display clb
*                    endif
*                  endif
 */
void CScrApp::trkrgt()
{
	if(fg_cobl) {
		putasy(clb_buf[clb_col]);
		ct_cstm++;	// increment count of chars to echo
	} else {
		incebf(&clb_col);
		chklm();
		dstrclb();
	}
}

/*--- function trkcr -----------------------------------
*         process a carriage return from the keyboard in terminal mode.
*
*         entry    none
*
*         exit     none
*
*         calls    scn2lnb to scan to last non-blank char
*                  trbotm  to send cursor to bottom line
*                  sendchs to send a character string to mainframe
*                  endrec  to terminate an output line
*                  kyereol to erase to end of line
*                  putasy  to output a character
*
*         method   if cursor is on bottom line then
*                    erase to end of line
*                    SEND A CR
*                  else if character was a carriage return
*                    transmit clb, omitting prompt, if any.  send cr.
*                  else
*                    transmit clb, including prompt, if any.  send cr.
*                  endif
 */
void CScrApp::trkcr()
{
   //ScrUtil.DebugOut("trkcr: fg_cobl=%d", fg_cobl);
   if(fg_cobl) {
		//!kyereol();
		endrec();
   } else {
		//! "xmit line" processing not implemented.
		int idxFirst=0;
		if(clb_fkc != MK_NOKYC) {
			// There is a computer-generated prompt.  Skip past it.
			idxFirst = clb_fkc;
		}
		// Send characters in clb_buf starting at idxFirst.
		Byte8 *pLWAP1 = clb_buf + LN_TXLEN;
		int nchars = scn2lnb(clb_buf+idxFirst, pLWAP1);
		sendchs(clb_buf+idxFirst, nchars);
		endrec();	// terminate with CR
		// if stay is off, SEND CURSOR TO BOTTOM LINE.
		if(!fg_stay) {
			trbotm();
		}
   }
}

/*--- function trkchar ---------------------------------
*         process an ascii character typed at the keyboard in
*         terminal mode.  this mostly applies to printing chars.
*
*         entry    m_inchar = the character
*
*         exit     none
*
*         calls    putasy  to output a character
*                  kyprint to process a character when not on bot line
*
*         method   if cursor is on bottom line then
*                    send char to mainframe
*                    increment count of chars sent
*                  else
*                    call kyprint to process char
*                  endif
 */
void CScrApp::trkchar()
{
   if(fg_cobl) {
      putasy(m_inchar); // send char to mainframe
      ct_cstm++;  // increment # of characters sent
   } else {
      // cursor is not on bottom line.  do local processing instead.
      kyprint(m_inchar);
   }
}

/*--- function trrbs ----------------------
*         process a backspace from the mainframe in term mode
*
*         entry    none
*
*         exit     none
*
*         calls    decbnz  to decrement a byte
*                  dstrclb to display the clb
*                  dstribf to display the bottom line
*                  chklm   to check the left margin
*                  kydelch to delete the character at the cursor
*
*         method   if cursor is on bottom line
*                    move cursor to left
*                    if the backspace was destructive then
*                      delete the character the cursor just moved to
*                    endif
*                    check the left margin
*                    display the clb
*                  else
*                    move the bottom line pointer to the left
*                    display the bottom line
*                  endif
 */
void CScrApp::trrbs()
{
	if(fg_cobl) {
		decbnz(&clb_col);	// decrement cursor location
		// if this is a destructive backspace, delete the character
		// the cursor has just been moved to.
		if(ct_debs) {
			ct_debs--;
		}
		chklm();	// check the left margin
		dstrclb();
	} else {
		decbnz(&pt_inln);	// decrement the buffer pointer
		dstribf();
	}
}

/*--- function sendchs -------------------------
*         send a string of chars as if typed at keyboard)
*
*         in terminal mode, send a string of chars to the mainframe.
*         ACT AS IF THE USER TYPED THEM IN.
*
*         entry    hl = fwa of character string
*                  c = # of chars to send (not including cr)
*
*         exit     ct.sctm reflects the chars that have been sent.
*
*         calls    putasy  to send a char
*
*         method   while charcount <> 0 do
*                    send character
*                    increment count of chars sent to mainframe
*                    decrement char count
*                    move to next char
*                  endwhile */
void CScrApp::sendchs(Byte8 *buf, int nchars)
{
	for(int j=0; j<nchars; j++) {
		putasy(buf[j]);
		ct_cstm++;	// account for this char for echoing
	}
}

/*--- function trkpgbk -------------------------------
*         page backward one screen in terminal mode
*
*         the user has just hit the 'previous page' key in terminal
*         mode, so perform this function.
*
*         entry    none
*
*         exit     none
*
*         calls    findlcm to find a line
*                  clb2ibf to copy the clb to the bottom line buffer
*                  addibf  to add this bottom line to textblock
*                  int2clb to move a textblock line to the clb
*                  prvlnb  to skip back to the previous line
*                  complno to compare line numbers
*                  dsscrn  to redisplay the whole screen
*                  movelno to copy a line number
*
*         method   (*  If there's more than a screen's worth of
*                      text, process bottom line case if necessary
*                      and loop backwards, simultaneously skipping
*                      back the top-of-screen and current-line
*                      pointers by one page's worth of lines.
*                      Clean things up at the end.  *)
*
*                  if there is <= a screen's worth in terminal mode then
*                    return
*                  endif
*                  IF CURSOR IS ON BOTTOM LINE THEN
*                    copy clb to bot line & add to textblock
*                  endif
*                  temptop := textblock addr of top line
*                  tempclb := textblock addr of clb
*                  for lineskip := screensize downto 1 do
*                    temptop := previous_line(temptop)
*                    if no previous line exists then
*                      break
*                    endif
*                    tempclb := previous_line(tempclb)
*                  endfor
*                  make temptop the top line of screen
*                  if tempclb does not point to clb then
*                    cursor_on_bottom_line := false
*                    move temptop^ to the clb
*                  endif
*                  redisplay screen */
void CScrApp::trkpgbk()
{
   // if there's only a screen or less in terminal mode, there's
   // nothing to do.
   if(ct_trln <= SC_PHSCN) {
      return;
   }
   //  backup the clb in the bottom line buffer if the cursor
   //  is on the bottom line.  (if it's not, we can't do this,
   //  because the bottom line buffer is being used for the
   //  bottom line.  anyway, there'd be no reason to.)
   if(fg_cobl) {
      clb2ibf();
      addibf();
   }
   // initialize 'temptop' to point to top line on screen
   Byte8 *pBlkIndexEntry, *pLineInBlock;
   findlcm(ds_tpln, pBlkIndexEntry, pLineInBlock);
   ss_pgtl = pLineInBlock;
   ss_pgti = pBlkIndexEntry;

   // Initialize temp clb to point to textblock copy of CLB.
   findlcm(clb_lno, pBlkIndexEntry, pLineInBlock);
   ss_pgcl = pLineInBlock;
   ss_pgci = pBlkIndexEntry;

   ss_pgct = SC_PHSCN-1;   // # of lines to skip.

   //  loop through one screen's worth of lines, bumping
   //  ss.pgti & ss.pgtl backwards each time to point to the
   //    previous candidate top line, and
   //  ss.pgci & ss.pgcl to point to the previous candidate clb.
   //  ss.pgct holds the # of lines left to skip.
   do {
      pLineInBlock = ss_pgtl;
      pBlkIndexEntry = ss_pgti;
      if(prvlnb(pBlkIndexEntry, pLineInBlock)) {
         ss_pgtl = pLineInBlock;
         ss_pgti = pBlkIndexEntry;

         // skip clb candidate pointer back one line.
         pLineInBlock = ss_pgcl;
         pBlkIndexEntry = ss_pgci;
         prvlnb(pBlkIndexEntry, pLineInBlock);
         ss_pgcl = pLineInBlock;
         ss_pgci = pBlkIndexEntry;
      }
   } while(--ss_pgct);

   // Make 'temptop' the new top line
   movelno(ss_pgtl, ds_tpln);

   if(complno(ss_pgcl, clb_lno)) {
      // we have a new clb, so set it up.
      fg_cobl = FALSE;  // no longer on bot line
      int2clb(ss_pgcl); // make this the clb
   }
   dsscrn();
}

/*--- function trkpgfr ------------------------------
*         process the 'page forward' key in terminal mode
*
*         entry    none
*
*         exit     none
*
*         calls    findtsc to find the top line on base screen
*                  complno to compare line numbers
*                  int2clb to put a line into the clb
*                  trbotm  to move to the bottom page
*                  nxtlnb  to move to the next line
*                  findlcm to find a line
*                  dsscrn  to redisplay the screen
*
*         method   (*  The basic idea behind this routine is a loop
*                      that skips both the top-of-screen pointer and
*                      the current-line pointer ahead by one page's
*                      worth of lines.  We must be careful to stop
*                      skipping when the top-of-screen pointer gets
*                      to within a screen's worth of the last line
*                      in memory.
*                      When the loop terminates, we move pointers
*                      around as determined by the outcome of the
*                      loop.  *)
*
*                  if cursor is not on bottom line then
*                    lasttop := the last acceptable top line
*                    tempclb := textblock addr of clb
*                    temptop := textblock addr of top line
*                    linect := # of lines in a screen
*                    while (linect > 0) and (temptop<>lasttop) do
*                      temptop := next_line(temptop)
*                      tempclb := next_line(tempclb)
*                      decrement linect
*                    endwhile
*                    if tempclb = bottom line then
*                      move cursor to bottom line
*                    else if tempclb <> the current clb then
*                      move tempclb^ to the clb
*                    endif
*                    make "temptop" the top line on screen
*                    redisplay the screen
*                  endif
 */
void CScrApp::trkpgfr()
{
   if(!fg_cobl) {
      // find and save last acceptable top line
      Byte8 *pBlkIndexEntry, *pLineInBlock;
      findtsc(pBlkIndexEntry, pLineInBlock);
      movelno(pLineInBlock, ln_pgbt);

      // Initialize temp clb to point to textblock copy of CLB.
      findlcm(clb_lno, pBlkIndexEntry, pLineInBlock);
      ss_pgcl = pLineInBlock;
      ss_pgci = pBlkIndexEntry;

      // initialize 'temptop' to point to top line on screen
      findlcm(ds_tpln, pBlkIndexEntry, pLineInBlock);
      ss_pgtl = pLineInBlock;
      ss_pgti = pBlkIndexEntry;

      ss_pgct = SC_PHSCN-1;   // # of lines to skip.

      // loop for one page's worth of lines (unless we hit the last
      // acceptable top line first), advancing the potential new
      // clb and top-of-screen pointers.
      while(ss_pgct--) {
         if(complno(ss_pgtl, ln_pgbt) >= 0) {
            break;
         }
         nxtlnb(ss_pgti, ss_pgtl);  // skip to next potential top line
         nxtlnb(ss_pgci, ss_pgcl);  // skip to next potential current line
      }

      // if the clb is now at the bottom line, use trbotm to send
      // us to the bottom line.
      if(0==complno(ss_pgcl, ln_bttr)) {
         trbotm();
      }
      // if the clb pointer no longer points to the current line,
      // make the new line the clb.
      if(complno(ss_pgcl, clb_lno)) {
         int2clb(ss_pgcl);
      }
      // now, set the new top line and redisplay the screen.
      movelno(ss_pgtl, ds_tpln);
      dsscrn();
   }
}

//=====  Communications  ============================================

/*--- function putasy ------------------------------
 *  Send a character to the mainframe.
 *  Entry:  inbyte   is the byte to send.
 *  //! This routine needs to be smarter about non-blocking I/O.
 */
void CScrApp::putasy(Byte8 inbyte)
{
   m_TelnetC.sendbyte(inbyte);
}

//===================================================================

void CScrApp::DebugDump(const char *msg)
{
   ScrUtil.DebugOut("== Debug dump %s", msg);
   ScrUtil.DebugOut("Dumping terminal mode textblock index");
   Byte8 *pBlkIndexEntry, *pLineInBlock;
   int iblk, nBlocksDumped=0;
	char szbuf[32];
	// Get list of textblocks
	std::string strBlockNums;
   for(iblk=0, pBlkIndexEntry = tridxpt; *pBlkIndexEntry != MK_INEND; 
      iblk++, nxtidx(pBlkIndexEntry)) {
		nBlocksDumped++;
      Byte8 BlockNum = pBlkIndexEntry[LN_LENNO];
		itoa(BlockNum, szbuf, 10);
		if(strBlockNums.length() > 0) strBlockNums += " ";
		strBlockNums += szbuf;
		if(iblk > 254) {
			ScrUtil.DebugOut("DebugDump: too many: %s", strBlockNums.c_str());
			CALL_SYSERR("DebugDump: too many blocks");
		}
	}
	ScrUtil.DebugOut("There are %d term blocks: %s", nBlocksDumped, strBlockNums.c_str());

	nBlocksDumped = 0;
   for(iblk=0, pBlkIndexEntry = tridxpt; *pBlkIndexEntry != MK_INEND; 
      iblk++, nxtidx(pBlkIndexEntry)) {
      nBlocksDumped++;
      char ASCII[24];
      memset(ASCII, 0, sizeof(ASCII));
      lno2asc(pBlkIndexEntry, ASCII);
      ScrUtil.DebugOut("  Index entry %2d: first line %s; %2d lines", iblk, ASCII, pBlkIndexEntry[OF_INNLN]);
      pLineInBlock = blkadr(pBlkIndexEntry);
      Byte8 BlockNum = pBlkIndexEntry[LN_LENNO];
      Word16 nActiveBytes = *((Word16 *) (pLineInBlock + BLK_OFFSET_ACTIVE_BYTES));
      ScrUtil.DebugOut("  Textblock %2d starts at %x; has %d bytes", BlockNum, pLineInBlock, nActiveBytes);
      pLineInBlock += BLK_HEAD;
      while(MK_BKEND != *pLineInBlock) {
         int len = pLineInBlock[OF_LNLEN];
         if(len < LN_HDLEN) {
            ScrUtil.DebugOut("** Error! len=%d, which is less than LN_HDLEN", len);
				CALL_SYSERR("DebugDump: bad line length");
            break;
         }
         lno2asc(pLineInBlock, ASCII);
         Byte8 rawtext[256];
         for(int j=0; j<(len-LN_HDLEN); j++) {
            Byte8 rawbyte = pLineInBlock[LN_HDLEN+j];
            rawtext[j] = isprint(rawbyte) ? rawbyte : '.';
         }
         rawtext[len-LN_HDLEN] = 0;
         ScrUtil.DebugOut("    Line at %x: %s=%s", pLineInBlock, ASCII, rawtext);

         // Skip to next line in block.
         pLineInBlock += len;
      }

   }
   ScrUtil.DebugOut("End of terminal mode textblock index; %d blocks", nBlocksDumped);

}

void CScrApp::InitializeMappings()
{
	Byte8 inchar;
	for(inchar=0x01; inchar <= 0x7f; inchar++) {
		EnterMappingHost(MD_TRM, inchar, FN_TR_R_CHAR);
      //BOOL bShift = (inchar >= 'A' && inchar <= 'Z');
      EnterMappingKybd(MD_TRM, FALSE, FALSE, FALSE, FALSE, inchar, FN_TR_K_CHAR);
	}
	EnterMappingHost(MD_TRM, '\b', FN_TR_R_BS);
	EnterMappingHost(MD_TRM, '\t', FN_TR_R_TAB);
	EnterMappingHost(MD_TRM, '\r', FN_TR_R_CR);
	EnterMappingHost(MD_TRM, '\n', FN_TR_R_LF);

   EnterMappingKybd(MD_TRM, FALSE, FALSE, FALSE, TRUE, VK_F2, FN_TR_K_DEBUG);
   EnterMappingKybd(MD_TRM, FALSE, FALSE, FALSE, TRUE, VK_F3, FN_TR_K_DEBUG2);
   EnterMappingKybd(MD_TRM, FALSE, FALSE, FALSE, FALSE, '\r', FN_TR_K_CR);
   EnterMappingKybd(MD_TRM, FALSE, FALSE, FALSE, FALSE, '\b', FN_TR_K_LFT);
   EnterMappingKybd(MD_TRM, FALSE, FALSE, FALSE, TRUE, VK_LEFT, FN_TR_K_LFT);
   EnterMappingKybd(MD_TRM, FALSE, FALSE, FALSE, TRUE, VK_RIGHT, FN_TR_K_RIGHT);
   EnterMappingKybd(MD_TRM, FALSE, FALSE, FALSE, TRUE, VK_UP, FN_TR_K_UP);
   EnterMappingKybd(MD_TRM, FALSE, FALSE, FALSE, TRUE, VK_DOWN, FN_TR_K_DOWN);
   EnterMappingKybd(MD_TRM, FALSE, FALSE, FALSE, TRUE, VK_PRIOR, FN_TR_K_PAGEUP);
   EnterMappingKybd(MD_TRM, FALSE, FALSE, FALSE, TRUE, VK_NEXT, FN_TR_K_PAGEDN);
	EnterMappingKybd(MD_TRM, FALSE, FALSE, FALSE, TRUE, VK_DELETE, FN_TR_K_DELCH);
	EnterMappingKybd(MD_TRM, FALSE, FALSE, FALSE, TRUE, VK_INSERT, FN_TOGGLE_INSERT);
}


void CScrApp::CallFuncForCode(TypFuncCode fncode)
{
	switch(fncode) {
   case FN_TR_K_DEBUG:
      DebugDump("from keypress");
      break;
   case FN_TR_K_DEBUG2:
      dsscrn();
      break;
   case FN_TR_R_CHAR:
      trrchar();
      break;
	case FN_TR_R_BS:
		trrbs();
		break;
	case FN_TR_R_TAB:
		trrtab();
		break;
   case FN_TR_R_CR:
      trrcr();
      break;
   case FN_TR_R_LF:
      trrlf();
      break;
   case FN_BEEP:
      do_nothing();
      break;
   case FN_SETESC:
      do_nothing();
      break;
   case FN_TR_K_CHAR:
      trkchar();
      break;
   case FN_TR_K_CR:
      trkcr();
      break;
   case FN_TR_K_LFT:
      trklft();
      break;
	case FN_TR_K_RIGHT:
		trkrgt();
		break;
   case FN_TR_K_UP:
      cursrup();
      break;
   case FN_TR_K_DOWN:
      trkcudn();
      break;
   case FN_TR_K_PAGEUP:
      trkpgbk();
      break;
   case FN_TR_K_PAGEDN:
      trkpgfr();
      break;
	case FN_TR_K_DELCH:
		kydelch();
		break;
	case FN_TOGGLE_INSERT:
		kyinsrt();
		break;
	}
}

BOOL CScrApp::CallFuncForMapKey(int key)
{
   BOOL bCalled=FALSE;
   TypFuncCodeMap::iterator iter = m_mappings.find(key);
   if(iter != m_mappings.end()) {
	   TypFuncCode fncode = iter->second;
	   CallFuncForCode(fncode);
      bCalled = TRUE;
   }
   return bCalled;
}

BOOL CScrApp::CallFuncForRcvChar(Byte8 inchar)
{
	m_inchar = inchar;
   int key = MAKE_KEY_FOR_RCV(fg_mode, inchar);
   return CallFuncForMapKey(key);
}

BOOL CScrApp::CallFuncForKybdChar(BOOL bShift, BOOL bControl, BOOL bAlt, 
   BOOL bIsFuncKey, Byte8 inchar)
{
	m_inchar = inchar;
   int key = MAKE_KEY_FOR_KBD(fg_mode, bShift, bControl, bAlt, bIsFuncKey, inchar);
   return CallFuncForMapKey(key);
}

/*--- function CScrApp::ProcByteFromHost ---------------------
 */
void CScrApp::ProcByteFromHost(unsigned char inbyte)
{
	if(fg_esc) {
		escape();
	} else if(fg_v2ec) {
		v2prec();
	} else {
		CallFuncForRcvChar(inbyte);
	}

}

BOOL CScrApp::StartConnect()
{
	BOOL bOK=FALSE;
	do {
		struct hostent * phe;
		unsigned long ipaddr=0;

		ipaddr = inet_addr(m_strHost.c_str());
		if(INADDR_NONE == ipaddr) {
			phe = gethostbyname(m_strHost.c_str());
			if(phe) {
				memcpy(&ipaddr, phe->h_addr_list[0], sizeof (unsigned long));
			} else {
				break;
			}
		}

		SOCKADDR_IN remoteAddr;

		m_TelnetC.m_sock = socket( AF_INET, SOCK_STREAM, 0 );

		memset(&remoteAddr, 0, sizeof (remoteAddr));
		remoteAddr.sin_family = AF_INET;
		remoteAddr.sin_port = htons(m_Port);
		memcpy(&remoteAddr.sin_addr.s_addr, &ipaddr, sizeof(remoteAddr.sin_addr.s_addr));
		//remoteAddr.sin_addr.s_addr = htonl(ipaddr);

#if 01
		// Set the socket to non-blocking.
		DWORD arg=1;
		ioctlsocket(m_TelnetC.m_sock, FIONBIO, &arg); 
		if (WSAAsyncSelect(m_TelnetC.m_sock, m_hwnd,
			 WM_SCR_MSG, FD_CONNECT | FD_READ | FD_CLOSE) == SOCKET_ERROR) {
			break;
		}
#endif
		if(connect(m_TelnetC.m_sock,(SOCKADDR *)&remoteAddr, sizeof(remoteAddr))) {
			// There's an error, but it's probably just WOULDBLOCK.
			DWORD dwLastErr = WSAGetLastError();
			if(WSAEWOULDBLOCK != dwLastErr) {
				// No, this is a real error.
			} else {
				bOK = TRUE;
			}
		} else {
			bOK= TRUE;
		}

	} while(false);
	return bOK;
}

void CScrApp::Disconnect()
{
   closesocket(m_TelnetC.m_sock);
   m_TelnetC.m_sock = 0;
	m_bConnected = FALSE;
	CauseRepaint();
}

void CScrApp::ProcessMacroChunk(HWND hWnd)
{
	if(m_MacroCharsSent >= m_MacroCurrent.length()) {
		KillTimer(hWnd, TIMER_MACRO);
	} else {
		BOOL bStop = FALSE;
		for(int j=m_MacroCharsSent; j<m_MacroCurrent.length() && !bStop; j++) {
			char ch = (Byte8) m_MacroCurrent[j];
			if('\\'==ch) {
				if(j+1 < m_MacroCurrent.length()) {
					j++;
					ch = m_MacroCurrent[j];
					if('r' == ch) {
						ch = '\r';
						bStop = TRUE;
					} else if('n' == ch) {
						ch = '\n';
						bStop = TRUE;
					}
				}
			}
			Byte8 thisbyte = (Byte8) ch;
			CallFuncForKybdChar(FALSE, FALSE, FALSE, FALSE, thisbyte);
		}
		m_MacroCharsSent = j;
	}
}

LRESULT CScrApp::OnCreate(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	m_hwnd = hWnd;
	int nArgsWithoutMinus=0;
	BOOL bCmdErr=FALSE;
	std::string strMacroNameStartup;
	for(char *pszArg=strtok((char *)m_strCmdLine.c_str(), " "); pszArg; pszArg=strtok(NULL, " ")) {
      if('-'==pszArg[0]) {
         std::string strName, strValue;
			CScrUtil::ParseArg(pszArg, ':', strName, strValue);
         if("-d" == strName) {
            ScrUtil.m_DbgLevel = atoi(strValue.c_str());
         } else if("-m" == strName) {
            strMacroNameStartup = strValue;
         } else {
				bCmdErr = TRUE;
			}
      } else {
         nArgsWithoutMinus++;
		   if(1==nArgsWithoutMinus) {
			   m_strHost = pszArg;
		   } else if(2==nArgsWithoutMinus) {
			   m_Port = atoi(pszArg);
		   }
      }
	}

	if(bCmdErr) {
		const char *pszUsageMsg = 
			"scredit2 - terminal emulator for SCOPE/Hustler\r\n"
			"http://scredit2.sourceforge.net\r\n"
			"\r\n"
			"Usage: scredit2 [host [port]] [-d:dbglevel] [-m:macname]\r\n"
			"where:\r\n"
			"dbglevel  is a debug level; 60 is highest\r\n"
			"macname   is a macro name to run at connect time\r\n"
			"\r\n"
			"Settings are read from Application Data\\scredit2.ini\r\n"
			;
		const char *pszUsageTitle = "scredit2 command line error";

		MessageBox(hWnd, pszUsageMsg, pszUsageTitle, MB_OK|MB_ICONHAND);
		PostQuitMessage(1);
		return 1;
	}
	if(strMacroNameStartup.length()) {
		m_MacroCurrent = m_Settings.GetMacroByName(strMacroNameStartup);
	}
	if(m_MacroCurrent.length()) SetTimer(hWnd, TIMER_MACRO, 2500, 0);

   SetTimer(hWnd, TIMER_CURSOR, 400, 0);
	if(m_strHost.length() > 0) {
		StartConnect();
	}
	return 0;
}

LRESULT CScrApp::OnSize(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	CauseRepaint();
	return 0;
}

LRESULT CScrApp::OnPaint(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	PAINTSTRUCT ps;
	HDC hdc;
	char szmsg[512];

	hdc = BeginPaint(hWnd, &ps);

	refBackground = RGB(0, 0, 0);

	HFONT hOldFont = NULL;
   if(!m_hFontNormal) {
      int nHeight = MulDiv(m_Settings.GetFontSize(), GetDeviceCaps(hdc, LOGPIXELSY), 72);
      m_hFontNormal = CreateFont(nHeight,0,0,0,0,0,0,0,0,0,0,0,
         FIXED_PITCH, m_Settings.GetFontFace());

		hOldFont = (HFONT)SelectObject(hdc, m_hFontNormal);

      TEXTMETRIC tm;
      GetTextMetrics(hdc,&tm);
      m_fontheight = tm.tmHeight + tm.tmExternalLeading;
      m_fontwidth = tm.tmAveCharWidth;
      m_MarginLeft = 2*m_fontwidth/3;
      m_MarginTop = m_fontheight/2;

		memset(szmsg, 'W', m_CharsPerLine);
		szmsg[m_CharsPerLine] = '\0';
		SIZE size;
		GetTextExtentPoint32(hdc, szmsg, m_CharsPerLine, &size);
		m_PixelsInLine = size.cx;

		m_penBorder = CreatePen(PS_SOLID, 1, refPenBorder);
   } else {
		hOldFont = (HFONT)SelectObject(hdc, m_hFontNormal);
	}

	RECT rect;
	GetClientRect(hWnd, &rect);
	HBRUSH hbrush;
	if(IsConnected()) {
		hbrush = (HBRUSH)GetStockObject(BLACK_BRUSH);
	} else {
		hbrush = (HBRUSH)GetStockObject(GRAY_BRUSH);
		refBackground = RGB(127,127,127);
	}

	SetTextColor(hdc, refForeground);
	SetBkColor(hdc, refBackground);

	int y = m_MarginTop;
	for(int iline=0; iline<m_LinesOnScreen; iline++) {
		BOOL bDrawWholeLine = TRUE;
		if(dp_cvrt == iline) {
			// This is the line that has the cursor on it.
			if(m_bCursorOn && dp_chor < SC_LNLEN) {
				bDrawWholeLine = FALSE;
				// Draw up to the cursor.
				TextOut(hdc, m_MarginLeft, y, (LPCSTR)screen+iline*SC_LNLEN, dp_chor);
				// Draw the cursor, which means invert the colors.
				SetTextColor(hdc, refBackground);
				SetBkColor(hdc, refForeground);
				TextOut(hdc, m_MarginLeft+dp_chor*m_fontwidth, y, (LPCSTR)screen+iline*SC_LNLEN+dp_chor, 1);
				// Draw the rest of the line.
				SetTextColor(hdc, refForeground);
				SetBkColor(hdc, refBackground);
				TextOut(hdc, m_MarginLeft+(1+dp_chor)*m_fontwidth, y, (LPCSTR)screen+iline*SC_LNLEN+dp_chor+1, SC_LNLEN-dp_chor-1);
			}
		}
		if(bDrawWholeLine) {
			TextOut(hdc, m_MarginLeft, y, (LPCSTR)screen+iline*SC_LNLEN, SC_LNLEN);
		}
#if 0
		// This is not necessary as long as we assume that we'll be drawing
		// all the way out to column 80.
		rect.left = m_MarginLeft + m_PixelsInLine;
		rect.top = y;
		rect.right = 9999;
		rect.bottom = rect.top + m_fontheight;
		FillRect(hdc, &rect, hbrush);
#endif
		y += m_fontheight;
	}

   // Finish by clearing the inactive region of the screen; that is,
   // the region to the left, right, top, and bottom of the text.
	//
   // topM left                                  |
	// leftMTop text text text text   rightM left | rightM right
	// leftMTop text text text text               |
	//          botMtop                           |
	// -------------------------------------------+ rightM right
	// bottomM Bottom                             bottomM Bottom

	int xborder = 2*m_MarginLeft+m_PixelsInLine;
	int yborder = 2*m_MarginTop+m_LinesOnScreen*m_fontheight;

   // Fill left margin, top
   rect.left = 0;
   rect.right = m_MarginLeft;
   rect.top = 0;
   rect.bottom = yborder;
   FillRect(hdc,&rect,hbrush);

   // Fill right margin, left
	rect.left = m_MarginLeft+m_PixelsInLine;
	rect.right = xborder;
	rect.bottom = yborder;
	FillRect(hdc, &rect, hbrush);

	// Fill right margin, right
   rect.left = xborder+1;
   rect.right = 9999;
	rect.bottom = yborder+1;
   FillRect(hdc,&rect,hbrush);
   // Fill top margin, left
   rect.bottom = m_MarginTop;
   rect.left = 0;
	rect.right = xborder-1;
   FillRect(hdc,&rect,hbrush);
	// Fill bottom margin, top
	rect.right = xborder-1;
	rect.top = m_MarginTop+m_LinesOnScreen*m_fontheight;
	rect.bottom = yborder;
   FillRect(hdc,&rect,hbrush);
	// Fill the bottom margin between the box around the screen and the status line.
   rect.right = 9999;
   rect.top = yborder+1;
   rect.bottom = rect.top+m_MarginTop;
   FillRect(hdc,&rect,hbrush);

	// Draw the status line
	memset(szmsg, ' ', sizeof(szmsg));
	y =  rect.bottom;
	if(fg_insr) {
		strncpy(szmsg+OF_FGINS, "Insert", strlen("Insert"));
	}
	TextOut(hdc, 0, y, " ", 1);	// left margin
	TextOut(hdc, m_MarginLeft, y, szmsg, sizeof(szmsg));

   // Fill bottom margin, bottom
   rect.right = 9999;
   rect.top = y+m_fontheight;//yborder+1;
   rect.bottom = 9999;
   FillRect(hdc,&rect,hbrush);

	// Draw a border around the printing portion of the screen.
	// First, the blank area just outside the region.
	HPEN penOld = (HPEN)SelectObject(hdc, m_penBorder);
	MoveToEx(hdc, 0, yborder, NULL);
	LineTo(hdc, xborder, yborder);
	LineTo(hdc, xborder, 0);

	SelectObject(hdc, penOld);
   SelectObject(hdc, hOldFont);

	EndPaint(hWnd, &ps);
	return 0;
}

LRESULT CScrApp::OnSockMsg(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	if(!m_bSysErr) {
   SOCKET sock=wParam;
	WORD wFD = LOWORD(lParam);
   WORD err = WSAGETSELECTERROR(lParam);
	char szbuf[8192];
	int j;
   //ScrUtil.DebugOut("OnSockMsg wFD=%d err=%d", wFD, err);
	if(err) {
		m_bConnected = FALSE;
		CauseRepaint();
	} else {
		switch(wFD) {
		case FD_CLOSE:
			m_bConnected = FALSE;
			CauseRepaint();
			break;
		case FD_CONNECT:
			// Connect succeeded.
			// We don't need to do much.
			m_bConnected = TRUE;
			CauseRepaint();
			break;
		case FD_READ:
			m_inbytes_raw = recv(m_TelnetC.m_sock, (char *)m_inbuf_raw, sizeof(m_inbuf_raw), 0);
			if(m_inbytes_raw <= 0) {
			} else {
				m_n_inbytes = m_TelnetC.interpret_telnet(m_inbuf_raw, m_inbytes_raw, m_inbuf);
            char szhex[4+3*2048];
            int ihex=0;
				for(j=0; j<m_n_inbytes; j++) {
					szbuf[j] = m_inbuf[j];
               ihex += sprintf(szhex+ihex, "%-2.2x ", m_inbuf[j]);
				}
				szbuf[j] = '\0';
            szhex[ihex] = '\0';
				ScrUtil.DebugOut("Got %d bytes: %s (%s)", m_n_inbytes, szbuf, szhex);
            // Now print 
				for(j=0; j<m_n_inbytes; j++) {
					ProcByteFromHost(m_inbuf[j]);
				}

            //! Consider removing this call to dsscrn, as an optimization.
            if(m_n_inbytes) {
#if 0
               if(m_bSendCmdsAtStartup) {
                  //! for debugging.
                  static int nLogins=0;
                  nLogins++;
                  char *szFakeResp="";
                  if(1==nLogins) {
                     szFakeResp="jal\r\n";
                  } else if(2==nLogins) {
                     szFakeResp="lukey\r\n";
                  } else if(3==nLogins) {
                     szFakeResp="dir\r\n";
                  } else if(4==nLogins) {
                     szFakeResp="ls -l\r\n";
                  } else if(5==nLogins) {
						   szFakeResp="wm\r\n";
					   }
                  if(szFakeResp[0]) {
                     m_TelnetC.sendbytes((Byte8*)szFakeResp, (int)strlen(szFakeResp));
                  }
               }
#endif
               dsscrn();
	   	      CauseRepaint();
            }
			}
			break;
		}

	}
	}
	return 0;
}

LRESULT CScrApp::OnKeyDown(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
   Byte8 nVirtKey = (Byte8) wParam;
   BOOL bShift = (::GetKeyState(VK_SHIFT) < 0);
   BOOL bCtrl  = (::GetKeyState(VK_CONTROL) < 0);
   BOOL bAlt  = (0 != (lParam & (1<<29)));

   m_bKeyHandled = CallFuncForKybdChar(bShift, bCtrl, bAlt, TRUE, nVirtKey);
   //ScrUtil.DebugOut("OnKeyDown: nVirtKey=%d bShift=%d bCtrl=%d bAlt=%d m_bKeyHandled=%d", nVirtKey, bShift, bCtrl, bAlt, m_bKeyHandled);
	return 0;
}

LRESULT CScrApp::OnChar(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
   if(!m_bKeyHandled) {
      Byte8 CharCode = (Byte8) wParam;
      BOOL bShift = FALSE; //(::GetKeyState(VK_SHIFT) < 0);
      BOOL bCtrl  = FALSE; //(::GetKeyState(VK_CONTROL) < 0);
      BOOL bAlt  = (0 != (lParam & (1<<29)));
      m_bKeyHandled = CallFuncForKybdChar(bShift, bCtrl, bAlt, FALSE, CharCode);
      //ScrUtil.DebugOut("OnChar: Char=%c bShift=%d bCtrl=%d bAlt=%d m_bKeyHandled=%d", CharCode, bShift, bCtrl, bAlt, m_bKeyHandled);
   }
   return 0;
}

LRESULT CScrApp::OnTimer(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
   if(TIMER_CURSOR == wParam) {
		m_bCursorOn = !m_bCursorOn;
		if(dp_chor < SC_LNLEN) {
			// Invert the character cell that corresponds to the cursor.
			HDC hdc = GetDC(hWnd);
#if 0
			// Now paint the cursor, which is one inverse video character cell.
			// *** This old code worked, but the cursor looked ugly when the foreground
			//     and background colors were not inverses of each other.
			RECT rectCursor;
			rectCursor.left = m_MarginLeft + dp_chor * m_fontwidth;
			rectCursor.right = rectCursor.left + m_fontwidth;
			rectCursor.top = m_MarginTop + dp_cvrt * m_fontheight;
			rectCursor.bottom = rectCursor.top + m_fontheight;
			InvertRect(hDispContext, &rectCursor);
#endif
			HFONT hOldFont = (HFONT)SelectObject(hdc, m_hFontNormal);
			if(m_bCursorOn) {
				SetTextColor(hdc, refBackground);
				SetBkColor(hdc, refForeground);
			} else {
				SetTextColor(hdc, refForeground);
				SetBkColor(hdc, refBackground);
			}
			int y = m_MarginTop + dp_cvrt*m_fontheight;
			TextOut(hdc, m_MarginLeft+dp_chor*m_fontwidth, y, (LPCSTR)screen+dp_cvrt*SC_LNLEN+dp_chor, 1);
			SelectObject(hdc, hOldFont);
			ReleaseDC(hWnd, hdc);
		}
   } else if(TIMER_MACRO == wParam) {
		ProcessMacroChunk(hWnd);
	}
   return 0;
}

/*--- function DlgProcConnect -----------------------
 *  This static function is the dialog procedure for the Connect dialog.
 */
BOOL CALLBACK CScrApp::DlgProcConnect(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
   char szbuf[256];
	TypConnSetting conn;
	int port;
   switch(message) {
	case WM_INITDIALOG:
		conn = pThisApp->m_Settings.GetDefaultConn();
		SetWindowText(GetDlgItem(hwnd, IDC_EDIT_HOSTNAME), conn.conn_host.c_str());
		port = conn.conn_port;
		if(0==port) port=23;
		SetWindowText(GetDlgItem(hwnd, IDC_PORT), _itoa(port, szbuf, 10));
      // Apparently this necessary because of the DIALOGEX statement in the .rc file.
      SetFocus(GetDlgItem(hwnd, IDC_EDIT_HOSTNAME));  
		break;
	case WM_CLOSE:
		{
			EndDialog(hwnd,0);
			break;
		}
	case WM_COMMAND:
		switch(LOWORD(wParam)) {
         case IDOK:
            {
               GetWindowText(GetDlgItem(hwnd, IDC_EDIT_HOSTNAME), szbuf, sizeof(szbuf));
               pThisApp->m_strHost = szbuf;
               GetWindowText(GetDlgItem(hwnd, IDC_PORT), szbuf, sizeof(szbuf));
               pThisApp->m_Port = atoi(szbuf);
               if(0==pThisApp->m_Port) pThisApp->m_Port = 23;
					conn.conn_host = pThisApp->m_strHost;
					conn.conn_port = pThisApp->m_Port;
					pThisApp->m_Settings.SetDefaultConn(conn);
               EndDialog(hwnd, IDOK);
            }
            break;
         case IDCANCEL:
            EndDialog(hwnd, IDCANCEL);
            break;
		}
	}
	return 0;
}

LRESULT CScrApp::OnCommand(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
   switch (LOWORD(wParam)) {
   case ID_FILE_CONNECT:
      if(IDOK == DialogBox(hInst, MAKEINTRESOURCE(IDD_DIALOG_CONNECT), hWndMain,
         (DLGPROC)DlgProcConnect)) {
         Disconnect();
         StartConnect();
      }
      return 0;
   case ID_FILE_DISCONNECT:
      Disconnect();
      return 0;
   }
   return 0;
}


LRESULT CScrApp::WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
	case WM_CREATE:
		OnCreate(hWnd, message, wParam, lParam);
		break;
   case WM_COMMAND:
      OnCommand(hWnd, message, wParam, lParam);
      break;
	case WM_PAINT:
		return OnPaint(hWnd, message, wParam, lParam);
	case WM_SIZE:
		return OnSize(hWnd, message, wParam, lParam);
	case WM_SCR_MSG:
		OnSockMsg(hWnd, message, wParam, lParam);
		break;
   case WM_KEYDOWN:
      //ScrUtil.DebugOut("WM_KEYDOWN    wParam=%-4.4x lParam=%-4.4x", wParam, lParam);
      OnKeyDown(hWnd, message, wParam, lParam);
      break;
   case WM_SYSKEYDOWN:
      // This is called when the Alt key is down when another key is pressed.
      //ScrUtil.DebugOut("WM_SYSKEYDOWN wParam=%-4.4x lParam=%-4.4x", wParam, lParam);
      // Special-case Alt-F4, which should always close the window.
      if(VK_F4==wParam && (0 != (lParam & (1<<29)))) {
   		return DefWindowProc(hWnd, message, wParam, lParam);
      } else {
         OnKeyDown(hWnd, message, wParam, lParam);
      }
      break;
   case WM_CHAR:
      OnChar(hWnd, message, wParam, lParam);
      break;
   case WM_TIMER:
      OnTimer(hWnd, message, wParam, lParam);
      break;
	default:
      //ScrUtil.DebugOut("calling DefWindowProc for msg %-4.4xx wParam %xx", message, wParam);
		return DefWindowProc(hWnd, message, wParam, lParam);
   }
   return 0;
}
